/* ************************************************************************** */
/* DSP processing */
/* ************************************************************************** */
#include "system.h"
#include "dsp.h"

#define DMA_EVENT_ST1   0x01
#define DMA_EVENT_ST2   0x02
#define DMA_EVENT_ST3   0x04
#define DMA_EVENT_ST4   0x08

#define _DMA_EVENT_set(event)    dmaEvents |= (event)
#define _DMA_EVENT_clr(event)    dmaEvents &= ~(event)
#define _DMA_EVENT_check(event)  (dmaEvents & (event))




/* Log Volume array with 70db range */
static const int aVolumeLog[101] = {
    0, 368044, 398933, 432414, 468705, 508042, 550681, 596898, 646993, 701294, 
    760151, 823948, 893100, 968055, 1049300, 1137365, 1232821, 1336287, 1448438, 1570001, 
    1701766, 1844590, 1999401, 2167204, 2349091, 2546243, 2759941, 2991575, 3242648, 3514794, 
    3809780, 4129523, 4476101, 4851766, 5258960, 5700328, 6178739, 6697302, 7259386, 7868644, 
    8529034, 9244850, 10020742, 10861752, 11773345, 12761445, 13832474, 14993391, 16251740, 17615699, 
    19094130, 20696641, 22433646, 24316433, 26357236, 28569318, 30967052, 33566022, 36383114, 39436637, 
    42746432, 46334008, 50222678, 54437713, 59006502, 63958736, 69326595, 75144962, 81451647, 88287632, 
    95697341, 103728923, 112434572, 121870858, 132099103, 143185773, 155202914, 168228615, 182347524, 197651389, 
    214239660, 232220133, 251709652, 272834867, 295733055, 320553018, 347456043, 376616957, 408225256, 442486342, 
    479622855, 519876121, 563507720, 610801184, 662063842, 717628817, 777857189, 843140343, 913902510, 990603527, 
    1073741824
};

/* Volume control srtucture */
static struct {
    int volumeL;	// volume value: min - 0, max - 0x40000000
    int volumeR;	// volume value: min - 0, max - 0x40000000
    int lfsr;		// LFSR random generate register
    int off;		// DC offset in 32-bit LSBs
}tVolCtrl;

/* Round control srtucture */
static struct {
    int snum;		// processing samples number
    int fqw;		// full quantize width (quantize width + 2)
    int qsh;		// quantize shift (30 - quantize width)
    int sash;		// saturation shift (32 - quantize width)
    int oash;		// out align shift (32 - data width)
    int obmsk;		// offset binary mask
    int shL;		// shape feedback value Left
    int shR;		// shape feedback value Left
    int lfsr1;		// LFSR random generate register
    int lfsr2;		// LFSR random generate register to triangle noise generation
}tRndCtrl; 

/* DMA control srtucture */
static struct {
    uint32_t ssa1;	// source start address 1
    uint32_t dsa1;	// destination start address 2
    uint32_t ssa2;	// source start address 2
    uint32_t dsa2;	// destination start address 2
    uint32_t ssa3;	// source start address 3
    uint32_t dsa3;	// destination start address 3
    uint32_t ssa4;	// source start address 4
    uint32_t dsa4;	// destination start address 4
}tDmaCtrl;

/* SDM control srtucture */
static struct {
    int acc1[2];
    int acc2[2];
    int acc3[2];
    int acc4[2];
    int acc5[2];
    int rMDO[2];
} tSdmCtrl;


// DSP processing FIFOs buffer
static int dspBuf[DSP_BUFFER_SIZE] __attribute__((coherent));

// ATT buffer pointer
static int attrptr;
// FIR stage1 buffer pointers
static int rptr1;
static int wptr1;
// FIR stage2 buffer pointers
static int rptr2;
static int wptr2;
// FIR stage3 buffer pointers
static int rptr3;
static int wptr3;
// FIR stage4 buffer pointers
static int rptr4;
static int wptr4;
// Round buffer pointer
static int rndptr;

// attenuator FIR buffer pointer
static int *pATT = &dspBuf[DSP_ATT_SECTOR_OFF];
// read FIR buffer pointers
static int *prFIR1 = &dspBuf[DSP_FIR1_SECTOR_OFF];
static int *prFIR2 = &dspBuf[DSP_FIR2_SECTOR_OFF];
static int *prFIR3 = &dspBuf[DSP_FIR3_SECTOR_OFF];
static int *prFIR4 = &dspBuf[DSP_FIR4_SECTOR_OFF];
// write FIR buffer pointers
static int *pwFIR1 = &dspBuf[DSP_FIR1_SECTOR_OFF + DSP_FIR1_CELL_SIZE];
static int *pwFIR2 = &dspBuf[DSP_FIR2_SECTOR_OFF + DSP_FIR2_CELL_SIZE];
static int *pwFIR3 = &dspBuf[DSP_FIR3_SECTOR_OFF + DSP_FIR3_CELL_SIZE];
static int *pwFIR4 = &dspBuf[DSP_FIR4_SECTOR_OFF + DSP_FIR4_CELL_SIZE];

// FIR output data round buffer
static int rndbuf[32];

// Output Data Buffer
static int dspOutputFIFO[OUTPUT_BUFFER_bSIZE/4] __attribute__((coherent));
// Num of pass FIR
static int dspFirPassNum;

// Zero value to cleare DSP FIFO buffers
static uint32_t dspZero __attribute__((coherent));
// DMA events control
static uint32_t dmaEvents;

/* Global variables */
volatile int dspOutPtr;


/******************************************************************************/
/************************ DSP assemlber functions *****************************/
/******************************************************************************/
extern void att32(void *pSrc, void *pDst, void *pVol);
extern void volume32(void *pSrc, void *pDst, void *pVol);

extern void fir1_5842(void *pSrc, void *pDst);
extern void fir2_5842(void *pSrc, void *pDst);
extern void fir3_5842(void *pSrc, void *pDst);

extern void fir1_44L(void *pSrc, void *pDst);
extern void fir1_44M(void *pSrc, void *pDst);
extern void fir1_88(void *pSrc, void *pDst);
extern void fir1_176(void *pSrc, void *pDst);
extern void fir1_352(void *pSrc, void *pDst);

extern void fir2(void *pSrc, void *pDst);
extern void fir3(void *pSrc, void *pDst);
extern void fir4(void *pSrc, void *pDst);

extern void dthrnd_f32(void *pSrc, void *pDst, void *pRnd);
extern void dthrnd_f16(void *pSrc, void *pDst, void *pRnd);
extern void shprnd_f32(void *pSrc, void *pDst, void *pRnd);
extern void shprnd_f16(void *pSrc, void *pDst, void *pRnd);

extern void sony5sdm_x8(void *pSrc, void *pDst, void *pSdm);
extern void sony5sdm_x4(void *pSrc, void *pDst, void *pSdm);
/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


/* Local functions */
static void FIR_44x8ovs (void);
static void FIR_44x16ovs (void);
static void FIR_88x4ovs (void);
static void FIR_88x8ovs (void);
static void FIR_176x2ovs (void);
static void FIR_176x4ovs (void);
static void FIR_352x2ovs (void);
static void FIR_bypass (void);

volatile static void (*callbackFIR)(void) = NULL;


/*
 * dsp_init() - return input attenuator buffer address
 *              to USB DMA data write
 */
uint32_t dsp_AttAdr_get(void)
{
    return KVA_TO_PA(pATT);
}

/*
 * dsp_init()
 */
void dsp_init (void)
{
        #ifdef OUT_OVS_x16
    tRndCtrl.snum = 16;     // set processing samples num
        #endif
        #ifdef OUT_OVS_x8
    tRndCtrl.snum = 8;      // set processing samples num
        #endif    
    
    // Init data to output round
    tRndCtrl.fqw = ROUND_DATA_WIDTH_bits + 2;       // set quantize shift (quantize width + 2)
    tRndCtrl.qsh = 30 - ROUND_DATA_WIDTH_bits;      // quantize shift (30 - quantize width)
    tRndCtrl.sash = 32 - ROUND_DATA_WIDTH_bits;     // saturation shift (32 - quantize width)
    tRndCtrl.oash = 32 - OUTPUT_DATA_WIDTH_bits;    // out data align shift (32 - data width)
    tRndCtrl.obmsk = OFFSET_BINARY << 31;           // set offset binary mask
    tRndCtrl.lfsr1 = DITHERING_SEED;                // set LFSR1 seed value
    tRndCtrl.lfsr2 = DITHERING_SEED * 100;          // set LFSR2 seed value
        
        #ifdef SYS_HOST_VOLUME_ENA
    tVolCtrl.volumeL = 0x40000000;
    tVolCtrl.volumeR = 0x40000000;
        #else
    tVolCtrl.volumeL = 0x40000000 * ATT_VALUE;
    tVolCtrl.volumeR = 0x40000000 * ATT_VALUE;
        #endif
    tVolCtrl.lfsr = DITHERING_SEED * 1000;
    tVolCtrl.off = DC_OFFSET_LSB << (31 - OUTPUT_DATA_WIDTH_bits);
    
    dspZero = 0;

    tDmaCtrl.ssa1 = KVA_TO_PA(&prFIR1[DSP_FIR1_FIFO_SIZE]);
    tDmaCtrl.dsa1 = KVA_TO_PA(prFIR1);
    tDmaCtrl.ssa2 = KVA_TO_PA(&prFIR2[DSP_FIR2_FIFO_SIZE]);
    tDmaCtrl.dsa2 = KVA_TO_PA(prFIR2);
    tDmaCtrl.ssa3 = KVA_TO_PA(&prFIR3[DSP_FIR3_FIFO_SIZE]);
    tDmaCtrl.dsa3 = KVA_TO_PA(prFIR3);
    tDmaCtrl.ssa4 = KVA_TO_PA(&prFIR4[DSP_FIR4_FIFO_SIZE]);
    tDmaCtrl.dsa4 = KVA_TO_PA(prFIR4);
    
    // Output data DMA address set
    _DMA_OutputSourceAddress_Set( KVA_TO_PA(dspOutputFIFO) );
}

/*
 * dsp_volume_set() - set volume in range from 0(mute) to 100(0db gain)
 */
void dsp_volume_set (uint16_t volume)
{
    tVolCtrl.volumeL = aVolumeLog[volume];
    tVolCtrl.volumeR = aVolumeLog[volume];
}

/*
    dsp_restart()
 */
void dsp_restart (eUSB_SAMPLING_FREQ sreq)
{
     switch (sreq)
     {
        case USB_SFREQ_44k:
        case USB_SFREQ_48k:
                #ifdef OUT_OVS_x16
            callbackFIR = FIR_44x16ovs;
                #endif
                #ifdef OUT_OVS_x8
            callbackFIR = FIR_44x8ovs;
                #endif
            dspFirPassNum = 1;
            break;

        case USB_SFREQ_88k:
        case USB_SFREQ_96k:
                #ifdef OUT_OVS_x16
            callbackFIR = FIR_88x8ovs;
                #endif
                #ifdef OUT_OVS_x8
            callbackFIR = FIR_88x4ovs;
                #endif
            dspFirPassNum = 2;
            break;

        case USB_SFREQ_176k:
        case USB_SFREQ_192k:
                #ifdef OUT_OVS_x16
            callbackFIR = FIR_176x4ovs;
                #endif
                #ifdef OUT_OVS_x8
            callbackFIR = FIR_176x2ovs;
                #endif
            dspFirPassNum = 4;
            break;
            
        case USB_SFREQ_352k:
        case USB_SFREQ_384k:
                #ifdef OUT_OVS_x16
            callbackFIR = FIR_352x2ovs;
                #endif
                #ifdef OUT_OVS_x8
            callbackFIR = FIR_bypass;
                #endif
            dspFirPassNum = 8;
            break;
            
        case USB_SFREQ_705k:
        case USB_SFREQ_768k:
            callbackFIR = FIR_bypass;           
             dspFirPassNum = 16;
            break;
            
        default: return;
    }
     
        #ifdef OUT_FRM_DSD64
    // Init DMA control structure
    tSdmCtrl.acc1[0] = 0;
    tSdmCtrl.acc1[1] = 0;
    tSdmCtrl.acc2[0] = 0;
    tSdmCtrl.acc2[1] = 0;
    tSdmCtrl.acc3[0] = 0;
    tSdmCtrl.acc3[1] = 0;
    tSdmCtrl.acc4[0] = 0;
    tSdmCtrl.acc4[1] = 0;
    tSdmCtrl.acc5[0] = 0;
    tSdmCtrl.acc5[1] = 0;
    tSdmCtrl.rMDO[0] = 0xE0000000;
    tSdmCtrl.rMDO[1] = 0xE0000000;
        #else
    // Cleare Shapers accumulators
    tRndCtrl.shL = 0;
    tRndCtrl.shR = 0;
        #endif

    // Init data pointers
    attrptr = DSP_ATT_SECTOR_SIZE/2;
    // Set read pointers to begin
    rptr1 = 0;
    rptr2 = 0;
    rptr3 = 0;
    rptr4 = 0;    
    // Set write pointers to middle
    wptr1 = DSP_FIR1_FIFO_SIZE/4;
    wptr2 = DSP_FIR2_FIFO_SIZE/4;
    wptr3 = DSP_FIR3_FIFO_SIZE/4;
    wptr4 = DSP_FIR4_FIFO_SIZE/4;

    // Cleare dma events
    dmaEvents = 0;
    // Cleare samples num
    _System_Num_Set(0);
    // Init output data pointer
    dspOutPtr = (OUTPUT_BUFFER_bSIZE/4)*3/4;
}

/*
    dsp_stop() - Cleare all dsp FIFO buffers
 */
void dsp_stop (void)
{
    while (_DMA_DSP_BusyCheck());
    _DMA_DSP_Start(KVA_TO_PA(&dspZero), sizeof(dspZero), KVA_TO_PA(dspBuf), sizeof(dspBuf));
}

/*******************************************************************************
 * FIR_44x8ovs() - FIR processing 
 * with x8 oversampling at 44.1/48kHz input sampling frequency
 ******************************************************************************/
void FIR_44x8ovs (void)
{
    // ************* Stage 1 FIR upsampler processing *****************
    fir1_5842( &prFIR1[rptr1], &pwFIR2[wptr2] );
    //fir1_44M( &prFIR1[rptr1], &pwFIR2[wptr2] );
    //fir1_44L( &prFIR1[rptr1], &pwFIR2[wptr2] );    
    // FIR read pointer increment & masking
    rptr1 += 2;
    rptr1 &= (DSP_FIR1_FIFO_SIZE-1);
    // FIR write pointer increment & masking
    wptr2 += 4;
    wptr2 &= (DSP_FIR2_FIFO_SIZE-1);
    
    if (wptr2 == 0) // Copy tail to head
        _DMA_EVENT_set(DMA_EVENT_ST2);
    
    // ************* Stage 2 FIR upsampler processing *****************
    fir2_5842( &prFIR2[rptr2], &pwFIR3[wptr3] );
    //fir2( &prFIR2[rptr2], &pwFIR3[wptr3] );
    // FIR read pointer increment & masking
    rptr2 += 4;
    rptr2 &= (DSP_FIR2_FIFO_SIZE-1);
    // FIR write pointer increment & masking
    wptr3 += 8;
    wptr3 &= (DSP_FIR3_FIFO_SIZE-1);
    
    if (wptr3 == 0) // Copy tail to head
        _DMA_EVENT_set(DMA_EVENT_ST3);
    
    // ************* Stage 3 FIR upsampler processing *****************
    fir3_5842( &prFIR3[rptr3], &rndbuf[rndptr] );
    //fir3( &prFIR3[rptr3], rndbuf );
    // FIR read pointer increment & masking
    rptr3 += 8;
    rptr3 &= (DSP_FIR3_FIFO_SIZE-1);
}

/*******************************************************************************
 * FIR_88x4ovs() - FIR processing 
 * with x4 oversampling at 88.2/96kHz input sampling frequency
 ******************************************************************************/
void FIR_88x4ovs (void)
{
    // ************* Stage 1 FIR upsampler processing *****************
    //fir1_5842( &prFIR1[rptr1], &pwFIR2[wptr2] );
    fir1_88( &prFIR1[rptr1], &pwFIR2[wptr2] );
    // FIR read pointer increment & masking
    rptr1 += 2;
    rptr1 &= (DSP_FIR1_FIFO_SIZE-1);    
    // FIR write pointer increment & masking
    wptr2 += 4;
    wptr2 &= (DSP_FIR2_FIFO_SIZE-1);
    
    if (wptr2 == 0) // Copy tail to head
        _DMA_EVENT_set(DMA_EVENT_ST2);
    
    // ************* Stage 2 FIR upsampler processing *****************
    //fir2_5842( &prFIR2[rptr2], &rndbuf[rndptr] );
    fir2( &prFIR2[rptr2], &rndbuf[rndptr] );
    // FIR read pointer increment & masking
    rptr2 += 4;
    rptr2 &= (DSP_FIR2_FIFO_SIZE-1);
    // round buffer poiner increment
    rndptr += 8;
}

/*******************************************************************************
 * FIR_176x2ovs() - FIR processing 
 * with x2 oversampling at 176.4/192kHz input sampling frequency
 ******************************************************************************/
void FIR_176x2ovs (void)
{
    // ************* Stage 1 FIR upsampler processing *****************
    fir1_176( &prFIR1[rptr1], &rndbuf[rndptr] );
    //fir1_352( &prFIR1[rptr1], &rndbuf[rndptr] );
    // FIR read pointer increment & masking
    rptr1 += 2;
    rptr1 &= (DSP_FIR1_FIFO_SIZE-1);
    // round buffer poiner increment
    rndptr += 4;
}


/*******************************************************************************
 * FIR_44x16ovs() - FIR processing 
 * with x16 oversampling at 44.1/48kHz input sampling frequency
 ******************************************************************************/
void FIR_44x16ovs (void)
{
    // ************* Stage 1 FIR upsampler processing *****************
    fir1_5842( &prFIR1[rptr1], &pwFIR2[wptr2] );
    //fir1_44M( &prFIR1[rptr1], &pwFIR2[wptr2] );
    // FIR read pointer increment & masking
    rptr1 += 2;
    rptr1 &= (DSP_FIR1_FIFO_SIZE-1);
    // FIR write pointer increment & masking
    wptr2 += 4;
    wptr2 &= (DSP_FIR2_FIFO_SIZE-1);
    
    if (wptr2 == 0) // Copy tail to head
        _DMA_EVENT_set(DMA_EVENT_ST2);
    
    // ************* Stage 2 FIR upsampler processing *****************
    fir2_5842( &prFIR2[rptr2], &pwFIR3[wptr3] );
    //fir2( &prFIR2[rptr2], &pwFIR3[wptr3] );
    // FIR read pointer increment & masking
    rptr2 += 4;
    rptr2 &= (DSP_FIR2_FIFO_SIZE-1);
    // FIR write pointer increment & masking
    wptr3 += 8;
    wptr3 &= (DSP_FIR3_FIFO_SIZE-1);
    
    if (wptr3 == 0) // Copy tail to head
        _DMA_EVENT_set(DMA_EVENT_ST3);
    
    // ************* Stage 3 FIR upsampler processing *****************
    fir3_5842( &prFIR3[rptr3], &pwFIR4[wptr4] );
    //fir3( &prFIR3[rptr3], &pwFIR4[wptr4] );
    // FIR read pointer increment & masking
    rptr3 += 8;
    rptr3 &= (DSP_FIR3_FIFO_SIZE-1);
    // FIR write pointer increment & masking
    wptr4 += 16;
    wptr4 &= (DSP_FIR4_FIFO_SIZE-1);
    
    if (wptr4 == 0) // Copy tail to head
        _DMA_EVENT_set(DMA_EVENT_ST4);
    
    // ************* Stage 4 FIR upsampler processing *****************
    fir4( &prFIR4[rptr4], &rndbuf[rndptr] );
    // FIR read pointer increment & masking
    rptr4 += 16;
    rptr4 &= (DSP_FIR4_FIFO_SIZE-1);
}

/*******************************************************************************
 * FIR_88x8ovs() - FIR processing 
 * with x8 oversampling at 88.2/96kHz input sampling frequency
 ******************************************************************************/
void FIR_88x8ovs (void)
{
    // ************* Stage 1 FIR upsampler processing *****************
    fir1_88( &prFIR1[rptr1], &pwFIR2[wptr2] );
    // FIR read pointer increment & masking
    rptr1 += 2;
    rptr1 &= (DSP_FIR1_FIFO_SIZE-1);
    // FIR write pointer increment & masking
    wptr2 += 4;
    wptr2 &= (DSP_FIR2_FIFO_SIZE-1);
    
    if (wptr2 == 0) // Copy tail to head
        _DMA_EVENT_set(DMA_EVENT_ST2);
    
    // ************* Stage 2 FIR upsampler processing *****************
    //fir2_5842( &prFIR2[rptr2], &pwFIR3[wptr3] );
    fir2( &prFIR2[rptr2], &pwFIR3[wptr3] );
    // FIR read pointer increment & masking
    rptr2 += 4;
    rptr2 &= (DSP_FIR2_FIFO_SIZE-1);
    // FIR write pointer increment & masking
    wptr3 += 8;
    wptr3 &= (DSP_FIR3_FIFO_SIZE-1);
    
    if (wptr3 == 0) // Copy tail to head
        _DMA_EVENT_set(DMA_EVENT_ST3);
    
    // ************* Stage 3 FIR upsampler processing *****************
    //fir3_5842( &prFIR3[rptr3], &rndbuf[rndptr] );
    fir3( &prFIR3[rptr3], &rndbuf[rndptr] );
    // FIR read pointer increment & masking
    rptr3 += 8;
    rptr3 &= (DSP_FIR3_FIFO_SIZE-1);
    // round buffer poiner increment
    rndptr += 16;
}

/*******************************************************************************
 * FIR_176x4ovs() - FIR processing 
 * with x4 oversampling at 176.4/192kHz input sampling frequency
 ******************************************************************************/
void FIR_176x4ovs (void)
{
    // ************* Stage 1 FIR upsampler processing *****************
    fir1_176( &prFIR1[rptr1], &pwFIR2[wptr2] );
    // FIR read pointer increment & masking
    rptr1 += 2;
    rptr1 &= (DSP_FIR1_FIFO_SIZE-1);    
    // FIR write pointer increment & masking
    wptr2 += 4;
    wptr2 &= (DSP_FIR2_FIFO_SIZE-1);
    
    if (wptr2 == 0) // Copy tail to head
        _DMA_EVENT_set(DMA_EVENT_ST2);
    
    // ************* Stage 2 FIR upsampler processing *****************
    fir2( &prFIR2[rptr2], &rndbuf[rndptr] );
    // FIR read pointer increment & masking
    rptr2 += 4;
    rptr2 &= (DSP_FIR2_FIFO_SIZE-1);
    // round buffer poiner increment
    rndptr += 8;
}

/*******************************************************************************
 * FIR_352x2ovs() - FIR processing 
 * with x2 oversampling at 352.8/384kHz input sampling frequency
 ******************************************************************************/
void FIR_352x2ovs (void)
{
    // ************* Stage 1 FIR upsampler processing *****************
    fir1_352( &prFIR1[rptr1], &rndbuf[rndptr] );
    // Input pointer increment & masking
    rptr1 += 2;
    rptr1 &= (DSP_FIR1_FIFO_SIZE-1);
    // round buffer poiner increment
    rndptr += 4;
}


/*******************************************************************************
 * FIR_bypass() - bypass FIR processing 
 * when input sampling freqency is equal to output sampling freqency
 ******************************************************************************/
void FIR_bypass (void)
{
    // Copy bypass data with -12db attenuation
    rndbuf[rndptr] = (prFIR1[rptr1] + 2)/4;
    rndbuf[rndptr+1] = (prFIR1[rptr1+1] + 2)/4;
    // FIR pointer increment & masking
    rptr1 += 2;
    rptr1 &= (DSP_FIR1_FIFO_SIZE-1);
    // round buffer poiner increment
    rndptr += 2;
}

    
/*******************************************************************************
 * dsp_proc() - FIR processing with input attenuation 
 * and output round with dithering and noise shaping
 ******************************************************************************/
void dsp_proc (void)
{
    int pass = dspFirPassNum;
    
    rndptr = 0;
    while (pass--)
    {
        /************************************************
         *  One Input Sample Processing         
         ************************************************/
        
        // Read one input sample and attenuate
        #ifdef SYS_HOST_VOLUME_ENA
            volume32(&pATT[attrptr], &pwFIR1[wptr1], &tVolCtrl);
        #else
            att32(&pATT[attrptr], &pwFIR1[wptr1], &tVolCtrl);
        #endif

        // Attenuator buffer pointer increment & masking
        attrptr += 2;
        attrptr &= (DSP_ATT_SECTOR_SIZE-1);    
        // FIR1 buffer write pointer increment & masking
        wptr1 += 2;
        wptr1 &= (DSP_FIR1_FIFO_SIZE-1);

        if (wptr1 == 0) // Copy tail to head
            _DMA_EVENT_set(DMA_EVENT_ST1);

        // Run one input sample FIR processing
        callbackFIR();

        // Copy FIR FIFO sectors tails to sector heads with DMA
        if (_DMA_DSP_BusyCheck() == 0) // if no busy dma channels
        {
            if (_DMA_EVENT_check(DMA_EVENT_ST4)) {
                _DMA_EVENT_clr(DMA_EVENT_ST4);
                _DMA_DSP_Start(tDmaCtrl.ssa4, DSP_FIR4_CELL_bSIZE, tDmaCtrl.dsa4, DSP_FIR4_CELL_bSIZE);
            }
            else if (_DMA_EVENT_check(DMA_EVENT_ST3)) {
                _DMA_EVENT_clr(DMA_EVENT_ST3);
                _DMA_DSP_Start(tDmaCtrl.ssa3, DSP_FIR3_CELL_bSIZE, tDmaCtrl.dsa3, DSP_FIR3_CELL_bSIZE);
            }
            else if (_DMA_EVENT_check(DMA_EVENT_ST2)) {
                _DMA_EVENT_clr(DMA_EVENT_ST2);
                _DMA_DSP_Start(tDmaCtrl.ssa2, DSP_FIR2_CELL_bSIZE, tDmaCtrl.dsa2, DSP_FIR2_CELL_bSIZE);
            }
            else if (_DMA_EVENT_check(DMA_EVENT_ST1)) {
                _DMA_EVENT_clr(DMA_EVENT_ST1);
                _DMA_DSP_Start(tDmaCtrl.ssa1, DSP_FIR1_CELL_bSIZE, tDmaCtrl.dsa1, DSP_FIR1_CELL_bSIZE);
            }
        }

        // Decrements samples num (bytes): one sample (4 bytes Left + 4 bytes Right)
        _System_Num_Dec(8);
    };
    
#ifdef OUT_FRM_PCM16
    // Output overflow and round processing 
        #ifdef NOISE_SHAPING_ENA
    shprnd_f16( rndbuf, &dspOutputFIFO[dspOutPtr], &tRndCtrl );
        #else
    dthrnd_f16( rndbuf, &dspOutputFIFO[dspOutPtr], &tRndCtrl );
        #endif
        #ifdef OUT_OVS_x8
    dspOutPtr += 8;
        #endif
        #ifdef OUT_OVS_x16
    dspOutPtr += 16;
        #endif
    dspOutPtr &= (OUTPUT_BUFFER_bSIZE/4-1);
#endif

#ifdef OUT_FRM_PCM32
        #ifdef NOISE_SHAPING_ENA
    shprnd_f32( rndbuf, &dspOutputFIFO[dspOutPtr], &tRndCtrl );
        #else
    dthrnd_f32( rndbuf, &dspOutputFIFO[dspOutPtr], &tRndCtrl );
        #endif
        #ifdef OUT_OVS_x8
    dspOutPtr += 16;
        #endif
        #ifdef OUT_OVS_x16
    dspOutPtr += 32;    
        #endif
    dspOutPtr &= (OUTPUT_BUFFER_bSIZE/4-1);
#endif

#ifdef OUT_FRM_DSD64
        #ifdef OUT_OVS_x8
    // Two pass SDM to 8 samples processing
    sony5sdm_x8(&rndbuf[0], &dspOutputFIFO[dspOutPtr], &tSdmCtrl);
    sony5sdm_x8(&rndbuf[8], &dspOutputFIFO[dspOutPtr+2], &tSdmCtrl);
        #endif
        #ifdef OUT_OVS_x16
    // Two pass SDM to 8 samples processing
    sony5sdm_x4(&rndbuf[0], &dspOutputFIFO[dspOutPtr], &tSdmCtrl);
    sony5sdm_x4(&rndbuf[16], &dspOutputFIFO[dspOutPtr+2], &tSdmCtrl);
        #endif
    // Output pointer increment & masking
    dspOutPtr += 4;
    dspOutPtr &= (OUTPUT_BUFFER_bSIZE/4-1);
#endif
}



