`include "DEFINES.v"

`define CMD_CON_VOLUME 		2'h0
`define CMD_CON_COEF 		2'h1



module DF_CONTROL
#( parameter
	MCLKSEL = `MCLK_SEL_x1024 // 0 - 1024Fs, 1 - 768Fs
)
(
/***** Global clock signals ********/
input iCLK, iCLRn,

// LRC input for automatic oversampling ratio selection
input iLRC,

/***** SPI control interface *******/
input iSCK, iDAT, iCS,
/***********************************/

/*** Coef RAM control outputs ******/
output oCWR, oCLKW,
output[10:0] oCADR,
output[11:0] oCDAT,
/***********************************/

/***** FIR config inputs ***********/
// attenuate control
input[1:0] iATT,		// 0: 0db, 1: -1db, 2: -2db, 3: -3db/manual value
// filter mode
input[1:0] iDF_MODE,	// 0 - normal, 1 - sharp, 2 - slow, 3 - short
// max oversampling ratio
input[2:0] iOVS_MAX,	// 0: x1 (bypass), 1 - x2, 2 - x4, 3 - x8, 4 - x16, 5 - x32
/***********************************/

/****** FIR control outputs ******/
output[2:0] oST_LAST,
// fir stages lengths ( LEN = coefNum - 1 )
output[`ST1_ADRW_BITS-1:0] oST1_LEN,
output[`ST2_ADRW_BITS-1:0] oST2_LEN,
output[`ST3_ADRW_BITS-1:0] oST3_LEN,
output[`ST4_ADRW_BITS-1:0] oST4_LEN,
output[`ST5_ADRW_BITS-1:0] oST5_LEN,
// fir stages types ( TYPE[0] - half band, TYPE[1] - minimal phase )
output[1:0] oST1_TYPE,
output[1:0] oST2_TYPE,
output[1:0] oST3_TYPE,
output[1:0] oST4_TYPE,
output[1:0] oST5_TYPE,
// coef stages addresses
output[10:0] oST1_COEF_ADR,
output[10:0] oST2_COEF_ADR,
output[10:0] oST3_COEF_ADR,
output[10:0] oST4_COEF_ADR,
output[10:0] oST5_COEF_ADR,
output[10:0] oVOLUME_ADR,
// Start FIR signal
output oSTART_FIR,
/***********************************/

// Input oversampling ratio
output[2:0] oINOVS	// 1 - x1 (44/48k), 2 - x2 (88/96k), 3 - x4 (176/192k), 4 - x8 (352/384k), 5 - x16 (705/768k),
);


/*********** SPI control interface ***************************/
reg rDC;			// Data - 1/Command - 0 flag
reg[4:0] rBCNT;
reg[10:0] rSH;
// COEF RAM control registers
reg rWR, rWREN;
reg[10:0] rADRCNT, wADRINIT;
// SPI CMD parse
wire wCMD_WRITE	= rSH[6];
wire[2:0] wCMD_CON = rSH[5:4];
wire[1:0] wCMD_FIR_TYPE = rSH[3:2];
// Custom FIR settings
reg[1:0] rFIRTYPE = `TYPE_ST1_HBS;
reg[7:0] rFIRLEN = `LEN_ST1_HBS;

/*********** Input sample rate detector **********************/
reg[1:0] rLRC;
reg[10:0] rSFCnt;
reg[7:0] rSF;
reg[2:0] rINOVS;
localparam SFLIM = (MCLKSEL == `MCLK_SEL_x1024) ? (128+63) : (96+47);

/*********** Start FIR signal generation *********************/
reg rStartFir;
reg[9:0] rStartCnt, rStartCntLim;
localparam SCNTLIM = (MCLKSEL == `MCLK_SEL_x1024) ? 1023 : 767;

/*********** Async FIR stages control ************************/
reg[`ST1_ADRW_BITS-1:0] wStg1Len /* synthesis syn_keep=1 */;
reg[1:0] wStg1Type;
reg[10:0] wStg1Adr /* synthesis syn_keep=1 */;
reg[10:0] wVolAdr;
wire[2:0] wOvsMax = (iOVS_MAX > `OVS_MAX_x32) ? `OVS_MAX_x32 : iOVS_MAX;

/************** Output Asigments *****************************/
assign oST_LAST = wOvsMax - rINOVS + `STIN_NUM;

assign oST1_LEN = wStg1Len;
assign oST2_LEN = `LEN_ST2_HB;
assign oST3_LEN = `LEN_ST3_HB;
assign oST4_LEN = `LEN_ST4_HB;
assign oST5_LEN = `LEN_ST5_HB;

assign oST1_TYPE = wStg1Type;
assign oST2_TYPE = `TYPE_ST2_HB;
assign oST3_TYPE = `TYPE_ST3_HB;
assign oST4_TYPE = `TYPE_ST4_HB;
assign oST5_TYPE = `TYPE_ST5_HB;

assign oST1_COEF_ADR = wStg1Adr;
assign oST2_COEF_ADR = `COEF_ADR_ST2_HB;
assign oST3_COEF_ADR = `COEF_ADR_ST3_HB;
assign oST4_COEF_ADR = `COEF_ADR_ST4_HB;
assign oST5_COEF_ADR = `COEF_ADR_ST5_HB;
assign oVOLUME_ADR = wVolAdr;

assign oINOVS = rINOVS;
assign oSTART_FIR = rStartFir;

assign oCWR = rWR;
assign oCLKW = iSCK;
assign oCADR = rADRCNT ^ 'h1;
assign oCDAT = {rSH, iDAT};
/*************************************************************/


/*************************************************************/
/*********** SPI control interface ***************************/
/*************************************************************/
always@ * begin
	case (wCMD_CON)
	`CMD_CON_VOLUME: wADRINIT = `COEF_ADR_ATT_CUST;
	`CMD_CON_COEF:   wADRINIT = `COEF_ADR_CUST;
	default: wADRINIT = 'h100;
	endcase
end

always@ (posedge iSCK or posedge iCS) begin
	if (iCS) begin
		rDC <= 0; rBCNT <= 0; rSH <= 0;
		rWR <= 0; rWREN <= 0; rADRCNT <= 0;
	end
	else begin
		rSH <= (rSH << 1) | iDAT;		
		
		if (!rDC) begin
			rBCNT <= (rBCNT == 7) ? 0 : (rBCNT + 1);
			
			if (rBCNT == 7) begin // if load cmd
				rDC <= 1;
				rADRCNT <= wADRINIT;
				rWREN <= wCMD_WRITE;
				if (wCMD_CON == `CMD_CON_COEF) rFIRTYPE <= wCMD_FIR_TYPE;
				if (wCMD_CON == `CMD_CON_COEF) rFIRLEN <= -1;
			end
		end
		else begin // if load data
			rBCNT <= (rBCNT == 23) ? 0 : (rBCNT + 1);			
			rWR <= (rBCNT == 10 || rBCNT == 22) ? ((rADRCNT < `COEF_ADR_ATT_CUST) ? 0 : rWREN) : 0;
			if (rWR) rADRCNT <= rADRCNT + 1;
			if (rWR) rFIRLEN <= rFIRLEN + rADRCNT[0];
		end
	end
end
/*************************************************************/


/*************************************************************/
/*********** Input sample rate detector **********************/
/*************************************************************/
always@ (posedge iCLK or negedge iCLRn) begin
	if (!iCLRn) begin
		rLRC <= 0;
		rSFCnt <= 0; rSF <= 0;
		rINOVS <= `INPUT_OVS_ERR;
		rStartFir <= 0; rStartCnt <= 0; rStartCntLim <= 0;
	end
	else begin
		rLRC <= (rLRC << 1) | iLRC;
		
		if (rLRC[0] && !rLRC[1]) begin
			rSFCnt <= 0;
			rSF <= rSFCnt >> 3;
		end
		else if (rSFCnt < 2047) begin
			rSFCnt <= rSFCnt + 1;
			
			// Input sample rate recognition
			if (rSF > SFLIM) rINOVS <= `INPUT_OVS_ERR;
			else if (rSF > (SFLIM/2)) rINOVS <= `INPUT_OVS_x1;
			else if (rSF > (SFLIM/4)) rINOVS <= `INPUT_OVS_x2;
			else if (rSF > (SFLIM/8)) rINOVS <= `INPUT_OVS_x4;
			else if (rSF > (SFLIM/16)) rINOVS <= `INPUT_OVS_x8;
			else if (rSF > (SFLIM/32)) rINOVS <= `INPUT_OVS_x16;
			else rINOVS <= `INPUT_OVS_ERR;
		end
		else
			rINOVS <= `INPUT_OVS_ERR;

		// Start FIR signal generation
		rStartCntLim <= (rINOVS == `INPUT_OVS_ERR) ? 0 : (SCNTLIM >> rINOVS);
		rStartCnt <= (rStartCnt == 0) ? rStartCntLim : (rStartCnt - 1);
		
		if (rStartCntLim == 0) rStartFir <= 0;
		else rStartFir <= (rStartCnt == 0) ? 1 : 0;
	end
end
/*************************************************************/


/*************************************************************/
/*********** Async FIR stages control ************************/
/*************************************************************/
always@ * begin
	
	if (MCLKSEL == `MCLK_SEL_x768) begin
		if (iDF_MODE == `DF_MODE_NORM) begin
			case (rINOVS)
			`INPUT_OVS_x1: begin wStg1Len = `LEN_ST1_HB; wStg1Type = `TYPE_ST1_HB; wStg1Adr = `COEF_ADR_ST1_HB; end
			`INPUT_OVS_x2: begin wStg1Len = `LEN_ST1_HB; wStg1Type = `TYPE_ST1_HB; wStg1Adr = `COEF_ADR_ST1_HB; end
			`INPUT_OVS_x4: begin wStg1Len = `LEN_ST1_HB; wStg1Type = `TYPE_ST1_HB; wStg1Adr = `COEF_ADR_ST1_HB; end
			`INPUT_OVS_x8: begin wStg1Len = `LEN_ST1_HBS; wStg1Type = `TYPE_ST1_HBS; wStg1Adr = `COEF_ADR_ST1_HBS; end
			`INPUT_OVS_x16: begin wStg1Len = `LEN_ST1_HBS; wStg1Type = `TYPE_ST1_HBS; wStg1Adr = `COEF_ADR_ST1_HBS; end
			default: begin wStg1Len = 'hx; wStg1Type = 'hx; wStg1Adr = 'hx; end
			endcase
		end
		else if (iDF_MODE == `DF_MODE_SHARP) begin
			case (rINOVS)
			`INPUT_OVS_x1: begin wStg1Len = `LEN_ST1_FIR; wStg1Type = `TYPE_ST1_FIR; wStg1Adr = `COEF_ADR_ST1_FIR; end
			`INPUT_OVS_x2: begin
					if (wOvsMax < `OVS_MAX_x16) begin wStg1Len = `LEN_ST1_FIR; wStg1Type = `TYPE_ST1_FIR; wStg1Adr = `COEF_ADR_ST1_FIR; end
					else 						begin wStg1Len = `LEN_ST1_FIRS; wStg1Type = `TYPE_ST1_FIRS; wStg1Adr = `COEF_ADR_ST1_FIRS; end
				end
			`INPUT_OVS_x4: begin wStg1Len = `LEN_ST1_FIRS; wStg1Type = `TYPE_ST1_FIRS; wStg1Adr = `COEF_ADR_ST1_FIRS; end
			`INPUT_OVS_x8: begin wStg1Len = `LEN_ST1_FIRS; wStg1Type = `TYPE_ST1_FIRS; wStg1Adr = `COEF_ADR_ST1_FIRS; end
			`INPUT_OVS_x16:	begin wStg1Len = `LEN_ST1_HBS; wStg1Type = `TYPE_ST1_HBS; wStg1Adr = `COEF_ADR_ST1_HBS; end
			default: begin wStg1Len = 'hx; wStg1Type = 'hx; wStg1Adr = 'hx; end
			endcase
		end
		else if (iDF_MODE == `DF_MODE_SHORT) begin
			case (rINOVS)
			`INPUT_OVS_x1: begin wStg1Len = `LEN_ST1_MPH; wStg1Type = `TYPE_ST1_MPH; wStg1Adr = `COEF_ADR_ST1_MPH; end
			`INPUT_OVS_x2: begin
					if (wOvsMax < `OVS_MAX_x8) begin wStg1Len = `LEN_ST1_MPH; wStg1Type = `TYPE_ST1_MPH; wStg1Adr = `COEF_ADR_ST1_MPH; end
					else						begin wStg1Len = `LEN_ST1x2_MPH; wStg1Type = `TYPE_ST1x2_MPH; wStg1Adr = `COEF_ADR_ST1x2_MPH; end
				end
			`INPUT_OVS_x4: begin wStg1Len = `LEN_ST1x2_MPH; wStg1Type = `TYPE_ST1x2_MPH; wStg1Adr = `COEF_ADR_ST1x2_MPH; end
			`INPUT_OVS_x8: begin wStg1Len = `LEN_ST1x4_MPH; wStg1Type = `TYPE_ST1x4_MPH; wStg1Adr = `COEF_ADR_ST1x4_MPH; end
			`INPUT_OVS_x16: begin wStg1Len = `LEN_ST1x4_MPH; wStg1Type = `TYPE_ST1x4_MPH; wStg1Adr = `COEF_ADR_ST1x4_MPH; end
			default: begin wStg1Len = 'hx; wStg1Type = 'hx; wStg1Adr = 'hx; end
			endcase
		end
		else if (iDF_MODE == `DF_MODE_CUST) begin
			wStg1Len = rFIRLEN;
			wStg1Type = rFIRTYPE;
			wStg1Adr = `COEF_ADR_CUST;
		end
	end
	else if (MCLKSEL == `MCLK_SEL_x1024) begin
		if (iDF_MODE == `DF_MODE_NORM) begin
			case (rINOVS)
			`INPUT_OVS_x1: begin wStg1Len = `LEN_ST1_HB; wStg1Type = `TYPE_ST1_HB; wStg1Adr = `COEF_ADR_ST1_HB; end
			`INPUT_OVS_x2: begin wStg1Len = `LEN_ST1_HB; wStg1Type = `TYPE_ST1_HB; wStg1Adr = `COEF_ADR_ST1_HB; end
			`INPUT_OVS_x4: begin wStg1Len = `LEN_ST1_HB; wStg1Type = `TYPE_ST1_HB; wStg1Adr = `COEF_ADR_ST1_HB; end
			`INPUT_OVS_x8: begin wStg1Len = `LEN_ST1_HB; wStg1Type = `TYPE_ST1_HB; wStg1Adr = `COEF_ADR_ST1_HB; end
			`INPUT_OVS_x16: begin wStg1Len = `LEN_ST1_HBS; wStg1Type = `TYPE_ST1_HBS; wStg1Adr = `COEF_ADR_ST1_HBS; end
			default: begin wStg1Len = 'hx; wStg1Type = 'hx; wStg1Adr = 'hx; end
			endcase
		end
		else if (iDF_MODE == `DF_MODE_SHARP) begin
			case (rINOVS)
			`INPUT_OVS_x1: begin wStg1Len = `LEN_ST1_FIR; wStg1Type = `TYPE_ST1_FIR; wStg1Adr = `COEF_ADR_ST1_FIR; end
			`INPUT_OVS_x2: begin wStg1Len = `LEN_ST1_FIR; wStg1Type = `TYPE_ST1_FIR; wStg1Adr = `COEF_ADR_ST1_FIR; end
			`INPUT_OVS_x4: begin wStg1Len = `LEN_ST1_FIRS; wStg1Type = `TYPE_ST1_FIRS; wStg1Adr = `COEF_ADR_ST1_FIRS; end
			`INPUT_OVS_x8: begin wStg1Len = `LEN_ST1_FIRS; wStg1Type = `TYPE_ST1_FIRS; wStg1Adr = `COEF_ADR_ST1_FIRS; end
			`INPUT_OVS_x16: begin wStg1Len = `LEN_ST1_FIRS; wStg1Type = `TYPE_ST1_FIRS; wStg1Adr = `COEF_ADR_ST1_FIRS; end
			default: begin wStg1Len = 'hx; wStg1Type = 'hx; wStg1Adr = 'hx; end
			endcase
		end
		else if (iDF_MODE == `DF_MODE_SHORT) begin
			case (rINOVS)
			`INPUT_OVS_x1: begin wStg1Len = `LEN_ST1_MPH; wStg1Type = `TYPE_ST1_MPH; wStg1Adr = `COEF_ADR_ST1_MPH; end
			`INPUT_OVS_x2: begin wStg1Len = `LEN_ST1_MPH; wStg1Type = `TYPE_ST1_MPH; wStg1Adr = `COEF_ADR_ST1_MPH; end
			`INPUT_OVS_x4: begin wStg1Len = `LEN_ST1x2_MPH; wStg1Type = `TYPE_ST1x2_MPH; wStg1Adr = `COEF_ADR_ST1x2_MPH; end
			`INPUT_OVS_x8: begin wStg1Len = `LEN_ST1x4_MPH; wStg1Type = `TYPE_ST1x4_MPH; wStg1Adr = `COEF_ADR_ST1x4_MPH; end
			`INPUT_OVS_x16: begin wStg1Len = `LEN_ST1x4_MPH; wStg1Type = `TYPE_ST1x4_MPH; wStg1Adr = `COEF_ADR_ST1x4_MPH; end
			default: begin wStg1Len = 'hx; wStg1Type = 'hx; wStg1Adr = 'hx; end
			endcase
		end
		else if (iDF_MODE == `DF_MODE_CUST) begin
			wStg1Len = rFIRLEN;
			wStg1Type = rFIRTYPE;
			wStg1Adr = `COEF_ADR_CUST;
		end
	end	
	
	case (iATT)
	0: wVolAdr = `COEF_ADR_ATT_0db;
	1: wVolAdr = `COEF_ADR_ATT_1db;
	2: wVolAdr = `COEF_ADR_ATT_2db;
	3: wVolAdr = `COEF_ADR_ATT_CUST;
	endcase
end
/*************************************************************/


endmodule






















