/************************************
	DF2 1.0 alpha
************************************/

`include "DEFINES.v"
`include "DF_CONTROL.v"
`include "ADR_GEN.v"
`include "FIR_RAM.v"
`include "FIR_MAC.v"
`include "DATA_READ.v"



module DF2_FIR_CORE
#( parameter 
	DW = 27, 	// data bus width in bits
	CW = 24,	// coefficients width in bits
	AW = 36, 	// accumulator width in bits
	MCLKSEL = `MCLK_SEL_x1024, // 0 - 1024Fs, 1 - 768Fs
	UPSAMPLE = `DF2_UPSAMPLE_NONE,
	LENMIN = 8,
	NSHOR = 1
)
(
/***** Global clock signals *****/
input iCLK, iCLRn,

// LRC input for automatic oversampling ratio selection
input iLRC,

/***** SPI control interface *******/
input iSPI_SCK, iSPI_DAT, iSPI_CS,
/***********************************/

/***** DF control inputs ***********/
// attenuate control
input[1:0] iATT,		// 0: 0db, 1: -1db, 2: -2db, 3: -3db/manual value
// filter mode
input[1:0] iDF_MODE,	// 0 - normal, 1 - sharp, 2 - short, 3 - slow/custom
// max oversampling ratio
input[2:0] iOVS_MAX,	// 0: x1 (bypass), 1 - x2, 2 - x4, 3 - x8, 4 - x16, 5 - x32,
// Ouput data width
input[4:0] iODW,		// output data width in bits
input iSHPEN, iDITHEN,

/***** Input Data Bus ************/
input signed[DW-1:0] DL, DR,
/***** Output Data Bus ***********/
output signed[23:0] QL, QR,

output[2:0] oINOVS,
output oOVFL, oOVFR
);

/**************************** DF_CONTROL output wires ***************************/
// Coef RAM control outputs
wire wCON_CWR, wCON_CLKW;
wire[10:0] wCON_CADR;
wire[11:0] wCON_CDAT;
// ADR GEN control
wire[2:0] wST_LAST;
// fir stages lengths ( LEN = coefNum - 1 )
wire[`ST1_ADRW_BITS-1:0] wST1_LEN;
wire[`ST2_ADRW_BITS-1:0] wST2_LEN;
wire[`ST3_ADRW_BITS-1:0] wST3_LEN;
wire[`ST4_ADRW_BITS-1:0] wST4_LEN;
wire[`ST5_ADRW_BITS-1:0] wST5_LEN;
wire[1:0] wST1_TYPE, wST2_TYPE, wST3_TYPE, wST4_TYPE, wST5_TYPE; // fir stages types ( TYPE[0] - half band, TYPE[1] - minimal phase )
wire[10:0] wST1_COEF_ADR, wST2_COEF_ADR, wST3_COEF_ADR, wST4_COEF_ADR, wST5_COEF_ADR, wVOLUME_ADR; // coef stages addresses
wire wSTART_FIR; // Start FIR signal

/**************************** ADR_GEN output wires ******************************/
wire wSTART_LOAD; // start signal to start MAC multiply
wire wSUM_LOAD; // signal to sum data for symmetry FIR
wire wIN_MUX; // mux MAC input to read input data sample
wire[2:0] wSTG_WR; // stage num to finish accumulate
/***** Coef ROM control signals ****/
wire[10:0] wCOEF_ADR_RD; // Coef ROM read address
wire[8:0] wADR_RD; // RAM read address
wire[8:0] wADR_WR; // RAM write address
wire wWR; // RAM write cmd
wire[`OBUF_ADRW_BITS-1:0] wSYNC_POS; // Output Buffer offset to read sync

/**************************** FIR_RAM output wires ******************************/
/**** Signals to MAC ****************/
wire wRAM_MAC_LOAD, wRAM_MAC_SUM, wRAM_MAC_MUX;
wire[2:0] wRAM_MAC_STG;
wire wRAM_MAC_ENA;
/**** Data Signals ******************/
wire signed[DW-1:0] wRAM_QL, wRAM_QR; // data output to DATA_READ

/**************************** FIR_COEF_RAM output wires *************************/
wire[CW/2-1:0] wCOEF_D;

/**************************** _FIR_MAC_L output wires ***************************/
wire signed[DW-1:0] wMAC_QL;
wire [2:0] wMAC_STG;

/**************************** _FIR_MAC_R output wires ***************************/
wire signed[DW-1:0] wMAC_QR;

/**************************** DATA_READ output wires ****************************/
wire[8:0] wOUT_ADR_RD; // data read address
wire wOUT_RD;

/********************************************************************************/
/**************************** DF_CONTROL ****************************************/
/********************************************************************************/
DF_CONTROL #(
	.MCLKSEL( MCLKSEL )		// 0 - 1024Fs, 1 - 768Fs
) _DF_CONTROL
(
	/***** Global clock signals ********/
	.iCLK( iCLK ), .iCLRn( iCLRn ),

	// LRC input for automatic oversampling ratio selection
	.iLRC( iLRC ),
	
	/*** SPI control interface input ***/
	.iSCK( iSPI_SCK ), .iDAT( iSPI_DAT ), .iCS( iSPI_CS ),
	/*** Coef RAM control outputs ******/
	.oCWR( wCON_CWR ), .oCLKW( wCON_CLKW ),
	.oCADR( wCON_CADR ),
	.oCDAT( wCON_CDAT ),

	/***** FIR config inputs ***********/
	// attenuate control
	.iATT( iATT ),				// 0: 0db, 1: -1db, 2: -2db, 3: -3db/manual value
	// filter mode
	.iDF_MODE( iDF_MODE ),		// 0 - normal, 1 - sharp, 2 - slow, 3 - short
	// max oversampling ratio
	.iOVS_MAX( iOVS_MAX ),		// 0: x1 (bypass), 1 - x2, 2 - x4, 3 - x8, 4 - x16, 5 - x32

	/****** FIR control outputs ******/
	.oST_LAST( wST_LAST ),
	// fir stages lengths ( LEN = coefNum - 1 )
	.oST1_LEN( wST1_LEN ), .oST2_LEN( wST2_LEN ), .oST3_LEN( wST3_LEN ),
	.oST4_LEN( wST4_LEN ), .oST5_LEN( wST5_LEN ),
	// fir stages types ( TYPE[0] - half band, TYPE[1] - minimal phase )
	.oST1_TYPE( wST1_TYPE ), .oST2_TYPE( wST2_TYPE ), .oST3_TYPE( wST3_TYPE ),
	.oST4_TYPE( wST4_TYPE ), .oST5_TYPE( wST5_TYPE ),
	// coef stages addresses
	.oST1_COEF_ADR( wST1_COEF_ADR ), .oST2_COEF_ADR( wST2_COEF_ADR ), .oST3_COEF_ADR( wST3_COEF_ADR ),
	.oST4_COEF_ADR( wST4_COEF_ADR ), .oST5_COEF_ADR( wST5_COEF_ADR ), .oVOLUME_ADR( wVOLUME_ADR ),
	// Start FIR signal
	.oSTART_FIR( wSTART_FIR ),

	// Input oversampling ratio
	.oINOVS( oINOVS )			// 1 - x1 (44/48k), 2 - x2 (88/96k), 3 - x4 (176/192k), 4 - x8 (352/384k), 5 - x16 (705/768k),
);

/********************************************************************************/
/**************************** ADR_GEN *******************************************/
/********************************************************************************/
ADR_GEN _ADR_GEN
(
	/***** Global clock signals *****/
	.iCLK( iCLK ), .iCLRn( iCLRn ),
	// enable read address generate logic
	.iRENA( !wOUT_RD ),
	// enable write generate logic
	.iWENA( wRAM_MAC_ENA ),
	// start MAC operation & read data from RAM
	.iSTART( wSTART_FIR ),

	/******** FIR settings *************/
	.iST_LAST( wST_LAST ),
	// fir stages lengths ( LEN = coefNum - 1 )
	.iST1_LEN( wST1_LEN ), .iST2_LEN( wST2_LEN ), .iST3_LEN( wST3_LEN ),
	.iST4_LEN( wST4_LEN ), .iST5_LEN( wST5_LEN ),
	// fir stages types ( TYPE[0] - half band, TYPE[1] - minimal phase )
	.iST1_TYPE( wST1_TYPE ), .iST2_TYPE( wST2_TYPE ), .iST3_TYPE( wST3_TYPE ),
	.iST4_TYPE( wST4_TYPE ), .iST5_TYPE( wST5_TYPE ),
	// coef stages addresses
	.iST1_COEF_ADR( wST1_COEF_ADR ), .iST2_COEF_ADR( wST2_COEF_ADR ), .iST3_COEF_ADR( wST3_COEF_ADR ),
	.iST4_COEF_ADR( wST4_COEF_ADR ), .iST5_COEF_ADR( wST5_COEF_ADR ), .iVOLUME_ADR( wVOLUME_ADR ),
	
	/***** MAC control signals *********/
	// write stage num from MAC
	.iSTG_WR( wMAC_STG ),
	// start signal to start MAC multiply
	.oSTART_LOAD( wSTART_LOAD ), .oSUM_LOAD( wSUM_LOAD ), .oIN_MUX( wIN_MUX ),	
	.oSTG_WR( wSTG_WR ),	// stage num to finish accumulate

	/***** Coef ROM control signals ****/
	// Coef ROM read address
	.oCOEF_ADR_RD( wCOEF_ADR_RD ),

	/***** RAM control signals *********/
	// RAM read address
	.oADR_RD( wADR_RD ),
	// RAM write address
	.oADR_WR( wADR_WR ),
	// RAM write cmd
	.oWR( wWR ),

	// Output Buffer offset to read sync
	.oSYNC_POS( wSYNC_POS )
);

/********************************************************************************/
/**************************** FIR_RAM *******************************************/
/********************************************************************************/
FIR_RAM #( .W( DW ) ) _FIR_RAM
(
	/***** Global clock signals *****/
	.iCLK( iCLK ), .iCLRn( iCLRn ),

	/**** Control signals to FIR_RAM ****/
	// read signal from DATA_READ
	.iOUT_RD( wOUT_RD ),
	// adr read from output
	.iADR_OUT( wOUT_ADR_RD ),

	/**** Signals from ADR_GEN **********/
	// adr read from ADR_GEN
	.iADR_RD( wADR_RD ),
	// adr write 
	.iADR_WR( wADR_WR ),
	// write signal
	.iWR( wWR ),
	.iSTART_LOAD( wSTART_LOAD ), .iSUM_LOAD( wSUM_LOAD ), .iDMUX( wIN_MUX ),
	.iSTG_NUM( wSTG_WR ),

	/**** Signals to MAC ****************/
	.oMAC_LOAD( wRAM_MAC_LOAD ), .oMAC_SUM( wRAM_MAC_SUM ), .oMAC_MUX( wRAM_MAC_MUX ),
	.oMAC_STG( wRAM_MAC_STG ),
	.oMAC_ENA( wRAM_MAC_ENA ),

	/**** Data Signals ******************/
	// data input from MAC
	.DL( wMAC_QL ), .DR( wMAC_QR ),
	// data output to DATA_READ
	.QL( wRAM_QL ), .QR( wRAM_QR )
);

/********************************************************************************/
/**************************** FIR_COEF_RAM **************************************/
/********************************************************************************/
FIR_COEF_RAM #( .CW( CW/2 ), .AW( 11 ) ) _COEF_RAM // CW - Coefficients Width, AW - Address Width
(
	.iCLRn( iCLRn ),
	.iCLKR( iCLK ),				// read clock
	.iCLKW( wCON_CLKW ), 		// write clock
	// control input
	.iADR_RD( wCOEF_ADR_RD ),	// read address
	.iADR_WR( wCON_CADR ),		// write address
	.iWR( wCON_CWR ),			// write enable
	// data input
	.iD( wCON_CDAT ),
	// data output
	.oD( wCOEF_D )
);

/********************************************************************************/
/**************************** _FIR_MAC_L ****************************************/
/********************************************************************************/
FIR_MAC #( .DW( DW ), .CW( CW/2 ), .AW( AW ) ) _FIR_MAC_L
(
	/***** Global clock signals *****/
	.iCLK( iCLK ), .iCLRn( iCLRn ), .iENA( wRAM_MAC_ENA ),
	// load multiplier data & start multiply
	.iSTART_LOAD( wRAM_MAC_LOAD ), .iSUM_LOAD( wRAM_MAC_SUM ), .iMUX( wRAM_MAC_MUX ),
	// write stage num from ADR_GEN
	.iSTG( wRAM_MAC_STG ),

	/******** MAC data inputs ********/
	// MAC data input from RAM
	.iDRAM( wRAM_QL ),
	// MAC data from input module
	.iDIN( DL ),
	// MAC coef input from ROM
	.iC( wCOEF_D ),

	/******** RAM control data ********/
	// MAC data output
	.oD( wMAC_QL ),
	// MAC write satge to ADR_GEN
	.oSTG( wMAC_STG )
);

/********************************************************************************/
/**************************** _FIR_MAC_R ****************************************/
/********************************************************************************/
FIR_MAC #( .DW( DW ), .CW( CW/2 ), .AW( AW ) ) _FIR_MAC_R
(
	/***** Global clock signals *****/
	.iCLK( iCLK ), .iCLRn( iCLRn ), .iENA( wRAM_MAC_ENA ),
	// load multiplier data & start multiply
	.iSTART_LOAD( wRAM_MAC_LOAD ), .iSUM_LOAD( wRAM_MAC_SUM ), .iMUX( wRAM_MAC_MUX ),
	// write stage num from ADR_GEN
	.iSTG( wRAM_MAC_STG ),

	/******** MAC data inputs ********/
	// MAC data input from RAM
	.iDRAM( wRAM_QR ),
	// MAC data from input module
	.iDIN( DR ),
	// MAC coef input from ROM
	.iC( wCOEF_D ),

	/******** RAM control data ********/
	// MAC data output
	.oD( wMAC_QR ),
	// MAC write satge to ADR_GEN
	.oSTG(  )
);

/********************************************************************************/
/**************************** DATA_READ *****************************************/
/********************************************************************************/
DATA_READ #( 
	.W( DW ),
	.MCLKSEL( MCLKSEL ),		// 0 - 1024Fs, 1 - 768Fs
	.ATTSH( LENMIN ),
	.UP( UPSAMPLE ), 
	.NSHOR( NSHOR )
) _DATA_READ
(
	/***** Global clock signals *****/
	.iCLK( iCLK ), .iCLRn( iCLRn ),

	/******** Output Data control *******/
	.iOVS_MAX( iOVS_MAX ),
	.iODW( iODW ),
	.iSHPEN( iSHPEN ), .iDITHEN( iDITHEN ),

	/***** IO data bus ******************/
	// input data
	.DL( wRAM_QL ), .DR( wRAM_QR ),
	// output 24bit data
	.QL( QL ), .QR( QR ),

	/***** Data RAM control *************/
	// Signal to sync output fifo buffer
	.iSYNC_POS( wSYNC_POS ),
	// data read address
	.oADR_RD( wOUT_ADR_RD ),
	.oOUT_RD( wOUT_RD ),

	// overflow signals
	.oOVFL( oOVFL ), .oOVFR( oOVFR )
);



endmodule


