/***************************************************************/
/******** DF2 alpfa 1.2  Cyclone FPGA Example ******************/
/***************************************************************/

`include "SAI_INPUT.v"
`include "DF2_FIR_CORE.v"
`include "SAI_OUTPUT.v"

`define DF2_DATA_BUS_WIDTH_BITS			27	// 27 or 31 recommended
`define DF2_COEF_BUS_WIDTH_BITS			24
`define DF2_MAC_WIDTH_BITS				36	// 36 or 40 recommended


module DF2
(
/***** Global clock signals **************/
input iCLK,

/***** SPI control input interface *******/
input iSPI_SCK, iSPI_DAT, iSPI_CS,

/***** DF control inputs *****************/
// attenuate control
input[1:0] iATT,		// 0: 0db, 1: -1db, 2: -2db, 3: -3db or custom value (load from SPI)
// filter mode
input[1:0] iDF_MODE,	// 0 - normal, 1 - sharp, 2 - short, 3 - slow or custom DF (load from SPI)
// output data round width
input[1:0] iODRW,		// Manual set output resolution: 0 - 8 bit, 1 - 12 bit, 2 - 16 bit, 3 - 20 bit
// oversampling control
input[1:0] iOVS_MAX,	// Manual set the maximum oversampling ratio
// DF ByPass Mode
input iBYPASS,			// 1 - turn ByPass Mode (output sample rate is always equal to the input sample rate)

// oscillator clock control
input iOSCSEL,			// OSC select input
output oOSC1EN, oOSC2EN,// Control outputs to enable selected OSC

/***** Oversampling & overflow indication ***********/
output[2:0] oINOVS,		// input sample rate indicate outputs: 0 - 44/48, 1 - 88/96, 2 - 176/192, 3 - 352/384, 4 - 705/768, 7 - error
output oCLIPP,			// Clipping input signal indicate output

/***** Serial data inputs ************/
input iLRC, iSCK, iDAT, iMUTE,
/***** Serial data outputs ***********/
output oDL, oDR, oBCK, oWCK, oDG,
output oMCKO			// Output masterclock to data source
);

/****************** Local Parameters ***************************/
localparam MCLKSEL = `MCLK_SEL_x1024;	// `MCLK_SEL_x1024  `MCLK_SEL_x768
localparam LENMIN = 8;					// Minimum output data width in bits (defines as minimum value of wODW)

/****************** SAI_INPUT output wires *********************/
wire signed[`DF2_DATA_BUS_WIDTH_BITS-1:0] wSAI_IN_QL, wSAI_IN_QR;
/****************** DF2_FIR_CORE output wires ******************/
wire signed[23:0] wFIR_CORE_QL, wFIR_CORE_QR;
wire[2:0] wINOVS;
wire wOVFL, wOVFR;

/***************************************************************/
/******** Optional Custom Control Example **********************/
/***************************************************************/
reg[4:0] wODW;							// Output data width in bits
reg[2:0] wOVS_MAX, wOVS_MAX_SET;		// `OVS_MAX_x1  `OVS_MAX_x2  `OVS_MAX_x4  `OVS_MAX_x8  `OVS_MAX_x16  `OVS_MAX_x32

// Async Logic
always@ * begin
	case (iODRW)	// output data round width control (decode manual setting)
	0: wODW = 8;
	1: wODW = 12;
	2: wODW = 16;
	3: wODW = 20;
	endcase
	
	case (iOVS_MAX)	// output oversampling control (decode manual setting)
	2'b11: wOVS_MAX_SET = `OVS_MAX_x1;
	2'b10: wOVS_MAX_SET = `OVS_MAX_x4;
	2'b01: wOVS_MAX_SET = `OVS_MAX_x8;
	2'b00: wOVS_MAX_SET = `OVS_MAX_x16;
	endcase

	// DF ByPass Processing
	if (wINOVS == `INPUT_OVS_ERR) wOVS_MAX = wOVS_MAX_SET; // Manual sampling frequency by default
	else if (iBYPASS) wOVS_MAX = wINOVS;				// Manual ByPass
	else if (wINOVS > wOVS_MAX_SET) wOVS_MAX = wINOVS;	// Automatic ByPass
	else wOVS_MAX = wOVS_MAX_SET;
end

// Counter Logic for overflow processing
reg[23:0] rOVFCNT;
reg rCLK, rOVF;

always@ (posedge iCLK) begin
	rCLK <= !rCLK;	// Divide iCLK frequency by 2
	
	// Counter to clipping indicate delay
	if (wOVFL | wOVFR)	rOVFCNT <= 11500000;
	else if (rOVFCNT) rOVFCNT <= rOVFCNT - rCLK;
	
	if (wOVFL | wOVFR) rOVF <= 1'b1;
	else if (!rOVFCNT) rOVF <= 1'b0;
end
/***************************************************************/
/***************************************************************/
/***************************************************************/

/*********** Output Assigments **********************/
assign oCLIPP = rOVF;
assign oMCKO = rCLK;		// F oMCK = F iCLK / 2
assign oOSC1EN = iOSCSEL;
assign oOSC2EN = !iOSCSEL;
assign oINOVS = wINOVS;
/****************************************************/


SAI_INPUT
#(
	.W( `DF2_DATA_BUS_WIDTH_BITS ),
	.FORMAT( `IFORMAT_I2S )				// `IFORMAT_I2S  `IFORMAT_LJ  `IFORMAT_RJ24  `IFORMAT_RJ16
) _SAI_INPUT
(
	// Mute control, active high
	.iMUTE( iMUTE ),
	// Serial data input
	.iLRC( iLRC ), .iSCK( iSCK ), .iDAT( iDAT ),
	// Parallel data output
	.QL( wSAI_IN_QL ), .QR( wSAI_IN_QR )
);


DF2_FIR_CORE 
#(
	.DW( `DF2_DATA_BUS_WIDTH_BITS ),
	.CW( `DF2_COEF_BUS_WIDTH_BITS ),
	.AW( `DF2_MAC_WIDTH_BITS ),
	.MCLKSEL( MCLKSEL ),				// 0 - 1024Fs, 1 - 768Fs
	.UPSAMPLE( `DF2_UPSAMPLE_NONE ),	// Additional UpSampling stage: `DF2_UPSAMPLE_x2 ... `DF2_UPSAMPLE_x32
	.LENMIN( LENMIN ),					// Minimum output data resolution (in bits)
	.NSHOR( 1 )							// Noise Shaper Order: 1..3, 0 - Shaper Off
) _DF2_FIR_CORE
(
	//***** Global clock signals ********
	.iCLK( iCLK ), .iCLRn( 1'b1 ),
	// LRC input for automatic oversampling ratio selection
	.iLRC( iLRC ),
	
	//***** SPI control interface *******
	.iSPI_SCK( iSPI_SCK ), .iSPI_DAT( iSPI_DAT ), .iSPI_CS( iSPI_CS ),

	//***** DF control inputs ***********
	// attenuate control
	.iATT( iATT ),						// 0: 0db, 1: -1db, 2: -2db, 3: -3db/manual value
	// filter mode
	.iDF_MODE( iDF_MODE ),				// 0: normal, 1: sharp, 2: short, 3: slow/custom
	// max oversampling ratio
	.iOVS_MAX( wOVS_MAX ),				// 0: x1 (bypass), 1 - x2, 2 - x4, 3 - x8, 4 - x16, 5 - x32
	.iODW( wODW ),						// output data width in bits
	.iSHPEN( 1'b1 ), .iDITHEN( 1'b1 ),	// Debug outputs to turn off shaping and dither

	//***** Input Data Bus **************
	.DL( wSAI_IN_QL ), .DR( wSAI_IN_QR ),
	//***** Output Data Bus *************
	.QL( wFIR_CORE_QL ), .QR( wFIR_CORE_QR ),

	.oINOVS( wINOVS ),					// Input sample rate indication 7 - error, 0 - 44/48, 1 - 88/96, 2 - 176/192, 3 - 352/384, 4 - 705/768
	.oOVFL( wOVFL ), .oOVFR( wOVFR )	// Input data overflow indication outputs
);


// Universal Series Audio Interface Output
USAI_OUTPUT
#(
	.MCLKSEL( MCLKSEL ),// Master Clock Select: 0 - 1024Fs, 1 - 768Fs
		
	.SPIMODE( 0 ),		// 0 - PCM, 1 - SPI
	.DDM( 0 ),			// Series Data mode enable
	
	// ********** Data Control **********************
	.DLEN( 20 ),		// Output Data Length in bits
	.OB( 0 ),			// 1 - offset Binary, 0 - 2's complement
	.DATINV( 0 ),		// Output serial data invertion	
	.HEADLEN( 0 ),		// Header Length in Bits
	.HEADL( 'h0 ),		// Header Value Left
	.HEADR( 'h0 ),		// Header Value Right
	// **********************************************
	
	// ********** Bit Clock Control *****************
	.BCKDIV( 1 ),		// BCK diveider: 0 - clk/1, 1 - clk/2, 2 - clk/3, 3 - clk/4, 4 - clk/6, 5 - clk/8
	.BCKINV( 0 ),		// 0 - update data on fall BCK, 1 - update data on rise BCK
	.CONT( 0 ),			// 1 - contineous bck enable	
	// **********************************************
	
	// ********** Word Clock Control ****************
	.WCKINV( 0 ),		// Wodr Clock inversion
	.LATINV( 0 ),		// Latch signal inversion
	// **********************************************
	
	// ********** Deglitcher Control ****************	
	.DGHW( 20 ),		// 2..30 - deglitcher HOLD width	
	.DGMODE( 0 ),		// 0 - Sync to WCK Rise, 1 - Sync to LAT Fall, 2 - Sync to WCK Fall
	.DGINV( 0 )			// 1 - invet DG strobe: 0 = 1-hold, 0-sample; 1 = 0-hold, 1-sample	
	// **********************************************
) _SAI_OUTPUT
(
	//***** Global clock signals **********
	.iCLK( iCLK ), .iCLRn( 1'b1 ),
	//***** Control inputs ****************
	.iOVS_MAX( wOVS_MAX ),	// 0: x1 (bypass), 1 - x2, 2 - x4, 3 - x8, 4 - x16, 5 - x32
	.iDGEN( 1'b1 ),		// 0 - turn deglitching signal to always sample state
	.iOUTZ( 1'b0 ),		// 1 - turn SPI outputs to Z-state
	//***** Parallel data bus inputs ******
	.iDL( wFIR_CORE_QL ), .iDR( wFIR_CORE_QR ),
	//***** Serial data outputs ***********
	.oDL( oDL ), .oDR( oDR ), .oBCK( oBCK ), .oWCK( oWCK ), .oLAT(  ), .oDG( oDG )
); // */





endmodule




















