
`include "DEFINES.v"
`include "UPSAMPLE.v"




module DATA_READ
#( parameter 
	W = 27,					// Input data Width
	MCLKSEL = `MCLK_SEL_x1024,		// 0 - 1024Fs, 1 - 768Fs
	ATTSH = 24,				// Defines as minimum output data resolution (in bits)
	NSHOR = 1,				// Noise Shape Order: 1..3, 0 - Shaper Off
	UP = `DF2_UPSAMPLE_NONE
)
(
/***** Global clock signals *********/
input iCLK, iCLRn,

/******** Output Data control *******/
input[2:0] iOVS_MAX,
input[4:0] iODW,
input iSHPEN, iDITHEN,
/************************************/

/***** IO data bus ******************/
// input data
input signed[W-1:0] DL, DR,
// output 24bit data
output signed[23:0] QL, QR,
/************************************/

/***** Data RAM control *************/
// Signal to sync output fifo buffer
input[`OBUF_ADRW_BITS-1:0] iSYNC_POS,
// data read address
output[8:0] oADR_RD,
output oOUT_RD,
/************************************/

// overflow signals
output oOVFL, oOVFR
);


//**** Input Control Logic ****************
wire[2:0] wOvsMax = (iOVS_MAX > `OVS_MAX_x32) ? `OVS_MAX_x32 : iOVS_MAX;
//*****************************************

//**** Control counter ********************
reg[1:0] rISEna;
reg[2:0] rOSEna;
reg[4:0] rUpCnt;
reg[9:0] rCnt;

wire[4:0] wUpCntLim = (1 << UP)-1;
wire[9:0] wCntLim = (MCLKSEL) ? (767 >> (wOvsMax+UP)) : (1023 >> (wOvsMax+UP));
wire wOSEna = (rCnt == 0) ? 1 : 0;
//*****************************************

//**** Read Address Logic *****************
reg rOutRead;
reg[8:0] rAdrRead;
reg[`OBUF_ADRW_BITS-1:0] rAdrCnt;
wire[`OBUF_ADRW_BITS-1:0] wAdrSync = (iSYNC_POS == rAdrCnt) ? (1 << (`OBUF_ADRW_BITS-1)) : 1;
//*****************************************

//**** Data Processing registers **********
reg rOVFL, rOVFR;
reg signed[W-1:0] wDL, wDR;
reg signed[W-2:0] wDINL, wDINR, rDINL, rDINR;
reg signed[23:0] wSHSUML, wSHSUMR, rSUML, rSUMR, rSH1L, rSH1R, rSH2L, rSH2R, rQL, rQR;
reg[23:0] rLFSR;
localparam LFSR_POLYNOM = 24'he81020;
//*****************************************

//**** Upsampler Outputs ******************
wire signed[W-1:0] wFIRL, wFIRR, wCICL, wCICR;
//*****************************************
// Round Value to output data width
wire signed[23:0] wRND = (iODW > 23) ? 0 : (1 << (23 - iODW));
//**** Async Dither & Shaping Logic *******
wire wLFSRCLRn = (!iCLRn) ? 0 : iDITHEN;
wire signed[23:0] wDITH = (ATTSH == 24) ? 0 : rLFSR; // No Dither for 24bit data
wire signed[23:0] wCALIBL, wCALIBR;
// Round data bus before upsampling
wire signed[24:0] wRNDL = (DL + (1<<(W-26))) >>> (W-25);
wire signed[24:0] wRNDR = (DR + (1<<(W-26))) >>> (W-25);
// Attenuate and round data before dithering and shaping (No Attenuation for 24bit data)
wire signed[23:0] wATTL = (ATTSH == 24) ? (rDINL >>> (W-25)) : ( ( rDINL - ((rDINL + (1 << (ATTSH-NSHOR-2))) >>> (ATTSH-NSHOR-1)) + (1<<(W-26)) ) >>> (W-25) );
wire signed[23:0] wATTR = (ATTSH == 24) ? (rDINR >>> (W-25)) : ( ( rDINR - ((rDINR + (1 << (ATTSH-NSHOR-2))) >>> (ATTSH-NSHOR-1)) + (1<<(W-26)) ) >>> (W-25) );
// Feed Back Error
	`ifdef LOGIC_LCMXO2
wire signed[23:0] wFBEL = iSHPEN * rSUML - iSHPEN * QL;
wire signed[23:0] wFBER = iSHPEN * rSUMR - iSHPEN * QR;
	`else
wire signed[23:0] wFBEL = (iSHPEN) ? (rSUML - QL) : 0;
wire signed[23:0] wFBER = (iSHPEN) ? (rSUMR - QR) : 0;
	`endif
// Round Attenuated Data and Add Shaping
wire signed[23:0] wSUML = wATTL + wSHSUML;
wire signed[23:0] wSUMR = wATTR + wSHSUMR;
// Summ dithering before quantize
wire signed[23:0] wQSUML = wSUML + (wDITH >>> iODW) + wRND;
wire signed[23:0] wQSUMR = wSUMR + (wDITH >>> iODW) + wRND;
// Mask to async quantizer control
wire[23:0] wMASK = (!iCLRn) ? 0 : (-1 << (24 - iODW));
//*****************************************

//**** Output Data Assigments *************
assign oOUT_RD = rOutRead;
assign oADR_RD = rAdrRead;
assign oOVFL = rOVFL;
assign oOVFR = rOVFR;
assign QL = rQL;
assign QR = rQR;
//*****************************************

always@ * begin
	// Upsampler select
	case (UP)
	`DF2_UPSAMPLE_NONE: begin wDL = DL; wDR = DR; end		// Process without upsampling
	`DF2_UPSAMPLE_x2:   begin wDL = wFIRL << (W-25); wDR = wFIRR << (W-25); end	// Process with FIR upsampling
	default:            begin wDL = wCICL << (W-25); wDR = wCICR << (W-25); end	// Process with CIC upsampling
	endcase
	// Overflow processing Left channel
	case (wDL[W-1:W-2])
	1: wDINL = {(W-1){1'b1}} >> 1;
	2: wDINL = 1 << (W-2);
	default: wDINL = wDL;
	endcase
	// Overflow processing Right channel
	case (wDR[W-1:W-2])
	1: wDINR = {(W-1){1'b1}} >> 1;
	2: wDINR = 1 << (W-2);
	default: wDINR = wDR;
	endcase
	// Choose Shaper Logic
	if (ATTSH == 24) begin		// No Shape for 24bit data
		wSHSUML = 0;
		wSHSUMR = 0;
	end
	else begin
		case (NSHOR)
		1: begin		// 1-st Order Shaper Logic
			wSHSUML = wFBEL;
			wSHSUMR = wFBER;
			end
		2: begin		// 2-nd Order Shaper Logic
			wSHSUML = (wFBEL << 1) - rSH1L;
			wSHSUMR = (wFBER << 1) - rSH1R;
			end
		3: begin		// 3-rd Order Shaper Logic
			wSHSUML = ((wFBEL - rSH1L) << 1) + (wFBEL - rSH1L) - ((wFBEL - rSH1L + 16) >>> 5) + rSH2L;
			wSHSUMR = ((wFBER - rSH1R) << 1) + (wFBER - rSH1R) - ((wFBER - rSH1R + 16) >>> 5) + rSH2R;
			end
		default: begin wSHSUML = 0; wSHSUMR = 0; end
		endcase
	end
end

/************* Generate white noise to dithering ****************/
always@ (posedge iCLK or negedge wLFSRCLRn) begin
	if (!wLFSRCLRn)	rLFSR <= 0;
	else if (rOSEna[2]) begin
		if (!rLFSR) rLFSR <= 1;
		else rLFSR <= (rLFSR >> 1) ^ (LFSR_POLYNOM * rLFSR[0]); // random generate
	end
end

/************* Data control processing **************************/
always@ (posedge iCLK or negedge iCLRn) begin
	if (!iCLRn) begin
		rISEna <= 0; rOSEna <= 0; rCnt <= 0; rUpCnt <= 0;
		rOutRead <= 0; rAdrCnt <= 0; rAdrRead <= 0;
		rOVFL <= 0; rOVFR <= 0;
		rDINL <= 0; rDINR <= 0;
		rSUML <= 0; rSUMR <= 0;
		rSH1L <= 0; rSH1R <= 0;
		rSH2L <= 0; rSH2R <= 0;
		rQL <= 0; rQR <= 0;
	end
	else begin
		// Control timer to read address generate
		rCnt <= (rCnt >= wCntLim) ? 0 : (rCnt + 1);
		if (rCnt >= wCntLim) rUpCnt <= (rUpCnt >= wUpCntLim) ? 0 : (rUpCnt + 1);
		rOutRead <= (!rCnt && !rUpCnt) ? 1 : 0;
		rISEna <= (rISEna << 1) | rOutRead;
		rOSEna <= (rOSEna << 1) | wOSEna;
		
		if (rISEna[1]) begin
			rAdrCnt <= rAdrCnt + wAdrSync;
			rAdrRead <= rAdrCnt | `OBUF_ADR_MASK;
		end
		
		if (rOSEna[2]) begin
			// Overflow Indicate Logic
			rOVFL <= wDL[W-1] ^ wDL[W-2];
			rOVFR <= wDR[W-1] ^ wDR[W-2];
			// Input Overflow Processing
			rDINL <= wDINL; rDINR <= wDINR;
			// Shaping Processing (Attenuate, Round, Dithering & Shaping)
			rSUML <= wSUML; rSUMR <= wSUMR;
			rSH1L <= wFBEL; rSH1R <= wFBER;
			rSH2L <= rSH1L; rSH2R <= rSH1R;
			// Output Quantizing
			rQL <= wQSUML & wMASK;
			rQR <= wQSUMR & wMASK;
		end
	end
end

/********* CIC Upsampler ******************/
CIC3 #(
	.W( 25 ), 		// Data width in bits
	.OVS( UP ) 	// Oversampling Ratio = 2 ^ OVS
) _CIC_L (
	.CLK( iCLK ), .CLRn( iCLRn ),
	.ISENA( rISEna[1] ), .OSENA( rOSEna[2] ),
	.D( wRNDL ),
	.Q( wCICL )
);
CIC3 #(
	.W( 25 ), 		// Data width in bits
	.OVS( UP ) 	// Oversampling Ratio = 2 ^ OVS
) _CIC_R (
	.CLK( iCLK ), .CLRn( iCLRn ),
	.ISENA( rISEna[1] ), .OSENA( rOSEna[2] ),
	.D( wRNDR ),
	.Q( wCICR )
);

/********* FIR x2 Upsampler ***************/
FIRx2 #(
	.W( 25 ) 		// Data width in bits
) _FIRx2_L (
	.CLK( iCLK ), .CLRn( iCLRn ),
	.ISENA( rISEna[1] ), .OSENA( rOSEna[2] ),
	.D( wRNDL ),
	.Q( wFIRL )
);
FIRx2 #(
	.W( 25 ) 		// Data width in bits
) _FIRx2_R (
	.CLK( iCLK ), .CLRn( iCLRn ),
	.ISENA( rISEna[1] ), .OSENA( rOSEna[2] ),
	.D( wRNDR ),
	.Q( wFIRR )
);


endmodule





