
`include "DEFINES.v"


module ADR_GEN
(
/***** Global clock signals *****/
input iCLK, iCLRn, 
// enable read address generate logic
input iRENA,
// enable write generate logic
input iWENA,
// start MAC operation & read data from RAM
input iSTART,

/******** FIR settings *************/
input[2:0] iST_LAST,
// fir stages lengths ( LEN = coefNum - 1 )
input[`ST1_ADRW_BITS-1:0] iST1_LEN,
input[`ST2_ADRW_BITS-1:0] iST2_LEN,
input[`ST3_ADRW_BITS-1:0] iST3_LEN,
input[`ST4_ADRW_BITS-1:0] iST4_LEN,
input[`ST5_ADRW_BITS-1:0] iST5_LEN,
// fir stages types ( TYPE[0] - half band, TYPE[1] - minimal phase )
input[1:0] iST1_TYPE,
input[1:0] iST2_TYPE,
input[1:0] iST3_TYPE,
input[1:0] iST4_TYPE,
input[1:0] iST5_TYPE,
// coef stages addresses
input[10:0] iST1_COEF_ADR,
input[10:0] iST2_COEF_ADR,
input[10:0] iST3_COEF_ADR,
input[10:0] iST4_COEF_ADR,
input[10:0] iST5_COEF_ADR,
input[10:0] iVOLUME_ADR,
/***********************************/

/***** MAC control signals *********/
// write stage num from MAC
input[2:0] iSTG_WR,
// start signal to start MAC multiply
output oSTART_LOAD,
// signal to sum data for symmetry FIR
output oSUM_LOAD,
// mux MAC input to read input data sample
output oIN_MUX,
// stage num to finish accumulate
output[2:0] oSTG_WR,
/***********************************/

/***** Coef ROM control signals ****/
// Coef ROM read address
output[10:0] oCOEF_ADR_RD,
/***********************************/

/***** RAM control signals *********/
// RAM read address
output[8:0] oADR_RD,
// RAM write address
output[8:0] oADR_WR,
// RAM write cmd
output oWR,
/***********************************/

// Output Buffer offset to read sync
output[`OBUF_ADRW_BITS-1:0] oSYNC_POS
);

// FIFO position registers for FIR stages
reg[`ST1_ADRW_BITS-1:0] rSt1_pos;
reg[`ST2_ADRW_BITS-1:0] rSt2_pos;
reg[`ST3_ADRW_BITS-1:0] rSt3_pos;
reg[`ST4_ADRW_BITS-1:0] rSt4_pos;
reg[`ST5_ADRW_BITS-1:0] rSt5_pos;
reg[`OBUF_ADRW_BITS-1:0] rOutBuf_pos;

/********************* Data read control ********************************/
// Initial data registers
reg rLenCorr;
reg[1:0] rStgType0, rStgType1;
reg[7:0] rRepeatNum, rStgLen, rCurrPos; 

// Address generate registers
reg[2:0] rStageCnt;
reg[2:0] rStgNum0, rStgNum1, rStgWr;
reg[7:0] rLen1, rLen2, rCnt, rOffset, rRptCnt;

// Output data registers
reg rStartLoad, rSumLoad0, rSumLoad1, rInputRead0, rInputRead1;
reg[8:0] rAdr0, rAdr1;

// Async Logic
wire[7:0] wCntInv = ~(rCnt >> 1) + !rRptCnt[0];
wire[7:0] wCnt = (rCnt[0]) ? (rCnt >> 1) : wCntInv;
wire[7:0] wSumAdr = wCnt - (rRptCnt >> 1) + rOffset;

/********************* Coef read control ********************************/
wire[8:0] wCoefCnt = {rCnt[7:1], rRptCnt[0], rCnt[0]};
reg[10:0] wCoefStartAdr;
reg[10:0] rCoefAdr0, rCoefAdr1, rCoefAdr2, rCoefAdr3;

/********************* Data write control *******************************/
// Wtire address register
reg rWr;
reg[8:0] wAdrWrite, rAdrWrite;


/************** Output Assigments for read control **********************/
// Wtire address to write
assign oADR_RD = rAdr1;
assign oSTART_LOAD = rStartLoad;
assign oSUM_LOAD = rSumLoad1;
assign oSTG_WR = rStgWr;
assign oIN_MUX = rInputRead1;
assign oSYNC_POS = rOutBuf_pos;
/************************************************************************/
/************** Output Assigments for Coef read control *****************/
// Coef read address
assign oCOEF_ADR_RD = rCoefAdr3;
/************************************************************************/
/************** Output Assigments for write control *********************/
// Wtire address to write
assign oADR_WR = rAdrWrite;
// Write signal to RAM
assign oWR = rWr;
/************************************************************************/


/************************************************************************/
/********************* Data read control ********************************/
/************************************************************************/
always@ (posedge iCLK or negedge iCLRn) begin
	if (!iCLRn)	rStageCnt <= 0;
	else if (iSTART) rStageCnt <= iST_LAST;
	else if (iRENA && !rRptCnt && rCnt >= rLen1) begin
		if (rStageCnt >= `STIN_NUM) rStageCnt <= rStageCnt - 1;
		else rStageCnt <= 0;
	end

	if (!iCLRn) begin
		// Initial data registers
		rRepeatNum <= 0; rStgLen <= 0; rCurrPos <= 0; rStgType0 <= 0; rStgType1 <= 0; rLenCorr <= 0;
		// Address generate registers		
		rStgNum0 <= 0; rStgNum1 <= 0; rStgWr <= 0;
		rLen1 <= 0; rLen2 <= 0; rCnt <= 0; rOffset <= 0; rRptCnt <= 0;
		// Output data registers
		rAdr0 <= 0; rAdr1 <= 0; rStartLoad <= 0; rSumLoad0 <= 0; rSumLoad1 <= 0; rInputRead0 <= 0; rInputRead1 <= 0;
	end
	else if (iRENA) begin

		if (rCnt < rLen1) begin
			rCnt <= rCnt + 1;
		end
		else begin
			rCnt <= 0;
			
			if (rRptCnt == 0) begin
				rRepeatNum <= 31 >> (`ST5_NUM - rStageCnt);
				
				if (rStageCnt == `STIN_NUM) begin // if load input data
					rStgLen <= 1; rLenCorr <= 0; rCurrPos <= 'hx; rStgType0 <= 'h0;
				end
				else begin // if fir processing
					case (rStageCnt)
					`ST1_NUM: begin rStgLen <= iST1_LEN; rLenCorr <= !iST1_LEN[0]; rCurrPos <= rSt1_pos; rStgType0 <= iST1_TYPE; end
					`ST2_NUM: begin rStgLen <= iST2_LEN; rLenCorr <= !iST2_LEN[0]; rCurrPos <= rSt2_pos; rStgType0 <= iST2_TYPE; end
					`ST3_NUM: begin rStgLen <= iST3_LEN; rLenCorr <= !iST3_LEN[0]; rCurrPos <= rSt3_pos; rStgType0 <= iST3_TYPE; end
					`ST4_NUM: begin rStgLen <= iST4_LEN; rLenCorr <= !iST4_LEN[0]; rCurrPos <= rSt4_pos; rStgType0 <= iST4_TYPE; end
					`ST5_NUM: begin rStgLen <= iST5_LEN; rLenCorr <= !iST5_LEN[0]; rCurrPos <= rSt5_pos; rStgType0 <= iST5_TYPE; end
					default: begin rStgLen <= 'h0;	 rLenCorr <= 0; 		   rCurrPos <= 'hx;	 rStgType0 <= 'hx; end
					endcase
				end
				
				rStgType1 <= rStgType0;			
				rStgNum0 <= rStageCnt;
				rStgNum1 <= rStgNum0;
				rRptCnt <= rRepeatNum;
				rLen1 <= rStgLen - rLenCorr;	// First inperpolate sample row lenght
				rLen2 <= (rStgType0 & `ST_TYPE_HALFBAND) ? 1 : (rStgLen + rLenCorr); // Second inperpolate sample row lenght
				rOffset <= (rStgType0 & `ST_TYPE_MIN_PHASE) ? (rCurrPos - 1) : (rCurrPos - (rStgLen >> 1) - 1); // Offset to center coefficient data position
			end
			else begin
				rRptCnt <= rRptCnt - 1;
				rLen1 <= rLen2;
				rLen2 <= rLen1;
			end
		end
		
		case (rStgNum1)
		`ST1_NUM: rAdr0 <= wSumAdr[`ST1_ADRW_BITS-1:0] | `ST1_ADR_MASK;
		`ST2_NUM: rAdr0 <= wSumAdr[`ST2_ADRW_BITS-1:0] | `ST2_ADR_MASK;
		`ST3_NUM: rAdr0 <= wSumAdr[`ST3_ADRW_BITS-1:0] | `ST3_ADR_MASK;
		`ST4_NUM: rAdr0 <= wSumAdr[`ST4_ADRW_BITS-1:0] | `ST4_ADR_MASK;
		`ST5_NUM: rAdr0 <= wSumAdr[`ST5_ADRW_BITS-1:0] | `ST5_ADR_MASK;
		default: rAdr0 <= 'hx;
		endcase
		rAdr1 <= rAdr0;
		
		// Control signals to MAC
		if (rCnt >= rLen1 && rLen1)	rStgWr <= (rStgNum1 == iST_LAST) ? `STOUT_NUM : (rStgNum1 + 1);
		else rStgWr <= 0;
		
		if (rStgType1 & `ST_TYPE_MIN_PHASE) rSumLoad0 <= 0;
		else if (rStgType1 & `ST_TYPE_HALFBAND) rSumLoad0 <= rCnt[0];
		else rSumLoad0 <= (!rRptCnt[0] && rCnt == 1) ? 0 : rCnt[0];
		rSumLoad1 <= rSumLoad0;
		
		rStartLoad <= rCnt[0];
		rInputRead0 <= (rStgNum1 == `STIN_NUM) ? 1 : 0;
		rInputRead1 <= rInputRead0;
	end
end

/************************************************************************/
/********************* Coef read control ********************************/
/************************************************************************/
always@ * begin
	if (rStgNum1 == (`ST1_NUM - 1))
		wCoefStartAdr = iVOLUME_ADR;
	else begin
		case (rStgNum1)
		`ST1_NUM: wCoefStartAdr = iST1_COEF_ADR;
		`ST2_NUM: wCoefStartAdr = iST2_COEF_ADR;
		`ST3_NUM: wCoefStartAdr = iST3_COEF_ADR;
		`ST4_NUM: wCoefStartAdr = iST4_COEF_ADR;
		`ST5_NUM: wCoefStartAdr = iST5_COEF_ADR;
		default: wCoefStartAdr = 'hx;		
		endcase
	end
end

always@ (posedge iCLK or negedge iCLRn) begin
	if (!iCLRn) begin
		rCoefAdr0 <= 0; rCoefAdr1 <= 0; rCoefAdr2 <= 0; rCoefAdr3 <= 0;
	end
	else if (iRENA) begin
		rCoefAdr0 <= wCoefStartAdr + wCoefCnt;
		rCoefAdr1 <= rCoefAdr0;		
		rCoefAdr2 <= rCoefAdr1;
		rCoefAdr3 <= rCoefAdr2;
	end
end

/************************************************************************/
/********************* Data write control *******************************/
/************************************************************************/
always@ * begin
	case (iSTG_WR)
	`ST1_NUM: wAdrWrite = rSt1_pos | `ST1_ADR_MASK;
	`ST2_NUM: wAdrWrite = rSt2_pos | `ST2_ADR_MASK;
	`ST3_NUM: wAdrWrite = rSt3_pos | `ST3_ADR_MASK;
	`ST4_NUM: wAdrWrite = rSt4_pos | `ST4_ADR_MASK;
	`ST5_NUM: wAdrWrite = rSt5_pos | `ST5_ADR_MASK;
	`STOUT_NUM: wAdrWrite = rOutBuf_pos | `OBUF_ADR_MASK;
	default: wAdrWrite = 'hx;
	endcase
end

always@ (posedge iCLK or negedge iCLRn) begin
	if (!iCLRn) begin
		rSt1_pos <= 0; rSt2_pos <= 0; rSt3_pos <= 0;
		rSt4_pos <= 0; rSt5_pos <= 0; rOutBuf_pos <= 0;
		rWr <= 0; rAdrWrite <= 0;
	end
	else if (iWENA) begin
		if (iSTG_WR == `ST1_NUM) rSt1_pos <= rSt1_pos + 1;
		if (iSTG_WR == `ST2_NUM) rSt2_pos <= rSt2_pos + 1;
		if (iSTG_WR == `ST3_NUM) rSt3_pos <= rSt3_pos + 1;
		if (iSTG_WR == `ST4_NUM) rSt4_pos <= rSt4_pos + 1;
		if (iSTG_WR == `ST5_NUM) rSt5_pos <= rSt5_pos + 1;
		if (iSTG_WR == `STOUT_NUM) rOutBuf_pos <= rOutBuf_pos + 1;
		
		rAdrWrite <= wAdrWrite;
		rWr <= (iSTG_WR) ? 1 : 0;
	end
end
/************************************************************************/


endmodule


















