
`include "DEFINES.v"

// Universal Series Audio Interface Output
module USAI_OUTPUT
#( parameter
	MCLKSEL = 0,	// Master Clock Select: 0 - 1024Fs, 1 - 768Fs
		
	SPIMODE = 1,	// 0 - PCM, 1 - SPI	
	DDM = 0,		// Dual Data mode (only for SPIMODE == 1) for series load dual channel SPI DAC
	
	// ********** Data Control **********************	
	DLEN = 20,		// Output Data Length in bits
	OB = 0,			// 1 - offset Binary, 0 - 2's complement
	DATINV = 0,		// Output data invertion
	HEADLEN = 4,	// Header Length in Bits
	HEADL = 'h1,	// Header Value Left
	HEADR = 'h1,	// Header Value Right
	// **********************************************
	
	// ********** Bit Clock Control *****************
	BCKDIV = 1,		// BCK diveider: 0 - clk/1, 1 - clk/2, 2 - clk/3, 3 - clk/4, 4 - clk/6, 5 - clk/8
	BCKINV = 0,		// 0 - update data on fall BCK, 1 - update data on rise BCK
	CONT = 0,		// 1 - contineous bck enable (only for SPIMODE == 0)	
	// **********************************************
		
	// ********** Word Clock Control ****************
	WCKINV = 0,		// Word Clock inversion
	LATINV = 0,		// Latch signal inversion
	// **********************************************
	
	// ********** Deglitcher Control ****************
	DGHW = 24,		// 1..31 - deglitcher HOLD width
	DGINV = 0,		// 1 - invet DG strobe: 0 = 1-hold, 0-sample; 1 = 0-hold, 1-sample
	DGMODE = 1		// 0 - Sync to WCK Rise, 1 - Sync to LAT Fall, 2 - Sync to WCK Fall
	// **********************************************
)
(
// input global signals
input iCLK, iCLRn,
// input control signals
input[2:0] iOVS_MAX,
input iDGEN,	// 1 - enable deglitcher signal
input iOUTZ,	// 1 - turn SPI outputs to Z-state
// input data
input [23:0] iDL, iDR,
// output data
output oDL, oDR, oBCK, oWCK, oLAT, oDG
);

// Insert logic gate delay
wire wKEEP_GND /* synthesis syn_keep=1 */ = 0;
wire wDELAY = (BCKDIV) ? 0 : wKEEP_GND;

//**** Input Data & Control Logic *********
reg [DLEN+HEADLEN-1:0] rSHL, rSHR;
wire[23:0] wDL = (DATINV) ? ~iDL : iDL;
wire[23:0] wDR = (DATINV) ? ~iDR : iDR;
wire[2:0] wOvsMax = (iOVS_MAX > `OVS_MAX_x32) ? `OVS_MAX_x32 : iOVS_MAX;
//*****************************************

//**** Control counter ********************
reg[9:0] rCNT;
wire[10:0] wMCKLIM = (MCLKSEL) ? 768 : 1024;
wire[9:0] wCntLim = (wMCKLIM >> wOvsMax) - 1;
reg[14:0] wDGLIM;
wire wDGCLR = (!iCLRn) ? 0 : iDGEN;
//*****************************************

// Control Registers
reg rDG, rWCK, rLAT, rSHIFT, rLOAD, rSTART, rREPEAT;
reg[1:0] rBCNT, rBCNTI, rBCK, rNBCK;
reg[2:0] rSCNT;
reg[9:0] rWCNT;

wire wDUAL_CLK = (BCKDIV == 0 || BCKDIV == 2) ? 1 : 0;
wire[9:0] wCKNUM = (wDUAL_CLK) ? (DLEN+HEADLEN) : ((DLEN+HEADLEN) << 1);
wire[9:0] wWCKLEN = 512 >> iOVS_MAX;
reg[9:0] wSTART;

//**** Output assigments *****************
assign oBCK = (iOUTZ) ? 1'bz : (rBCK[1] ^ rNBCK[1]);
assign oWCK = (iOUTZ) ? 1'bz : (rWCK ^ WCKINV ^ wDELAY);
assign oLAT = (CONT || !SPIMODE) ? 0 : (rLAT ^ LATINV ^ wDELAY);
assign oDG = rDG ^ DGINV ^ wDELAY;
assign oDL = (iOUTZ) ? 1'bz : (rSHL[DLEN+HEADLEN-1] ^ wDELAY);
assign oDR = (iOUTZ) ? 1'bz : ((DDM) ? 0 : (rSHR[DLEN+HEADLEN-1] ^ wDELAY));
//****************************************

// async logic
always@ * begin
	case (BCKDIV)
	0: rBCNTI = 0;
	1: rBCNTI = 0;
	2: rBCNTI = 2;
	3: rBCNTI = 1;
	4: rBCNTI = 2;
	5: rBCNTI = 3;
	default: rBCNTI = 2'hx;
	endcase
	
	if (CONT) wSTART = wCntLim - wCKNUM * (rBCNTI + 1) - 6;
	else if (SPIMODE == 0) wSTART = wCntLim - 5;
	else if (DGMODE == 0) wSTART = 0;
	else if (DGMODE == 1) wSTART = wCntLim - 1;
	else if (DGMODE == 2) wSTART = wCntLim - 5;

	case (wOvsMax)
	`OVS_MAX_x1: wDGLIM = ((wMCKLIM>>`OVS_MAX_x1)-1) * DGHW;
	`OVS_MAX_x2: wDGLIM = ((wMCKLIM>>`OVS_MAX_x2)-1) * DGHW;
	`OVS_MAX_x4: wDGLIM = ((wMCKLIM>>`OVS_MAX_x4)-1) * DGHW;
	`OVS_MAX_x8: wDGLIM = ((wMCKLIM>>`OVS_MAX_x8)-1) * DGHW;
	`OVS_MAX_x16: wDGLIM = ((wMCKLIM>>`OVS_MAX_x16)-1) * DGHW;
	`OVS_MAX_x32: wDGLIM = ((wMCKLIM>>`OVS_MAX_x32)-1) * DGHW;
	default: wDGLIM = 'hx;
	endcase
end

always@ (posedge iCLK or negedge iCLRn) begin
	if (!iCLRn) begin
		rCNT <= 0;
		rSHL <= 0; rSHR <= 0;
		rSCNT <= 0; rBCNT <= 0; rWCNT <= 0; rBCK <= 0; 		
		rWCK <= 0; rLAT <= 0; rSHIFT <= 0; rLOAD <= 0;
		rSTART <= 0; rREPEAT <= 0;
	end
	else begin
		// Control timer to read address generate
		rCNT <= (rCNT == wCntLim) ? 0 : (rCNT+1);
		
		if (rSTART) rSTART <= 0;
		else if (rCNT == wSTART) rSTART <= 1;
		else if (rWCNT == 1 && !rBCNT) rSTART <= rREPEAT;
		
		if (rSTART) rSCNT <= 7;
		else if (rSCNT) rSCNT <= rSCNT - 1;

		if (rSCNT == 3) rREPEAT <= !rREPEAT & DDM;
		
		// LAT Processing Logic
		if (rSCNT == 7) rLAT <= rREPEAT;
		else if (rSCNT == 5) rLAT <= 1;
		
		// Word Clock Processing Logic
		if (SPIMODE) begin
			if (rREPEAT) begin
				if (rSCNT == 6) rWCK <= 1;
				else if (rSCNT == 3) rWCK <= 0;
			end
			else begin
				if (rCNT == wSTART) rWCK <= 1;
				else if (rSCNT == 3) rWCK <= 0;
			end
		end
		else if (CONT) begin
			if (rCNT == 0) rWCK <= 0;
			else if (rCNT == (wCntLim/2+1)) rWCK <= 1;
		end
		else begin
			if (rWCNT == 4) rWCK <= 1;
			else if (rSCNT == 3) rWCK <= 0;
		end		
		
		if (rSCNT == 3) rBCNT <= rBCNTI;
		else if (rBCNT) rBCNT <= rBCNT - 1;
		else if (rWCNT | CONT) rBCNT <= rBCNTI;
		
		if (rSCNT == 3) rWCNT <= wCKNUM;
		else if ((rWCNT | CONT) && !rBCNT) rWCNT <= rWCNT - 1;

		rBCK[1] <= (rWCNT | CONT) ? (!rWCNT[0] ^ !BCKINV) : 0;
		rBCK[0] <= (rWCNT | CONT) ? !rWCNT[0] : 0;
		
		rLOAD <= (rSCNT == 3) ? !rREPEAT : 0;
		if ( (rSCNT == 3) || ( (rWCNT[0] | wDUAL_CLK) && !rBCNT ) ) rSHIFT <= (rWCNT > 0) ? 1 : 0;
		else rSHIFT <= 0;
		
		// Load input data and shift to output
		if (rLOAD) begin
			rSHL <= ((HEADL & {HEADLEN{1'b1}}) << DLEN) | ((wDL ^ (OB << 23)) >> (24-DLEN));
			rSHR <= ((HEADR & {HEADLEN{1'b1}}) << DLEN) | ((wDR ^ (OB << 23)) >> (24-DLEN));
		end
		else if (rSHIFT) begin
			rSHL <= (rSHL << 1) | (DDM & rSHR[DLEN+HEADLEN-1]);
			rSHR <= rSHR << 1;
		end
	end
end

//**** Deglitcher signal processing ****************
always@ (posedge iCLK or negedge wDGCLR) begin
	if (!wDGCLR) rDG <= 0;
	else if (rCNT == wCntLim-1) rDG <= 1;
	else if (rCNT == ((wDGLIM>>5)-1)) rDG <= 0;
end

//**** Negative clock sync processing **************
always@ (negedge iCLK or negedge iCLRn) begin
	if (!iCLRn)
		rNBCK <= 0;
	else begin
		rNBCK[0] <= rBCK[0];
	
		case (BCKDIV)
		0: rNBCK[1] <= rBCK[0];
		2: rNBCK[1] <= rNBCK[0];
		default: rNBCK[1] <= 0;
		endcase
	end
end


endmodule



// Simple Series Audio Interface Output
module SSAI_OUTPUT
#( parameter 
	MCLKSEL = 0,	// Master Clock Select: 0 - 1024Fs, 1 - 768Fs
	
	// ********** Data Control **********************
	DLEN = 24, 		// Output Data Length in bits
	OB = 0,			// 1 - offset binary, 0 - 2's complement
	DATINV = 0,		// Output data invertion
	// **********************************************

	// ********** Bit Clock Control *****************
	BCKDIV = 3, 	// BCK = iCLK/(2^CKDIV)
	BCKINV = 0,		// 1 - invet BCK clokck: 0 = update data on falling BCK; 1 = update data on rising BCK
	// **********************************************
	
	// ********** Word Clock Control ****************
	WCKW = 16,		// WCK strobe width in iCLK periods	
	WCKINV = 0,		// 1 - invet WCK strobe: 0 = update DAC on falling WCK; 1 = update DAC on rising WCK
	// **********************************************
	
	// ********** Deglitcher Control ****************
	DGHW = 20,		// 1..31 - deglitcher HOLD width
	DGINV = 0		// 1 - invet DG strobe: 0 = 1-hold, 0-sample; 1 = 0-hold, 1-sample
	// **********************************************
)
(
// input global signals
input iCLK, iCLRn,
// input control signals
input[2:0] iOVS_MAX,
input iDGEN,	// 1 - enable deglitcher signal
// input data
input [23:0] iDL, iDR,
// output data (LAT - not used)
output oDL, oDR, oBCK, oWCK, oLAT, oDG
);


//**** Input Control Logic ****************
wire[2:0] wOvsMax = (iOVS_MAX > `OVS_MAX_x32) ? `OVS_MAX_x32 : iOVS_MAX;
wire signed[23:0] wCoding = OB << 23;
wire wDGCLR = (!iCLRn) ? 0 : iDGEN;
//*****************************************

//**** Control counter ********************
reg [9:0] rCNT;
wire[10:0] wMCKLIM = (MCLKSEL) ? 768 : 1024;
wire[9:0] wCntLim = (wMCKLIM >> wOvsMax) - 1;
wire wBCK = (BCKDIV) ? rCNT[BCKDIV-1] : !iCLK;
wire wLOAD = (rCNT == ((wCntLim - (DLEN << BCKDIV)) & wCntLim)) ? 1 : 0;
//*****************************************

reg rBCK, rWCK, rDG;
reg[DLEN-1:0] rSHL, rSHR;
reg[14:0] wDGLIM;
wire[23:0] wDL = (DATINV) ? ~iDL : iDL;
wire[23:0] wDR = (DATINV) ? ~iDR : iDR;

//**** Output assigments *****************
assign oBCK = wBCK ^ BCKINV;
assign oWCK = rWCK ^ WCKINV;
assign oDG = rDG ^ DGINV;
assign oDL = rSHL[DLEN-1];
assign oDR = rSHR[DLEN-1];
//****************************************

//*****************************************
always@ * begin
	case (wOvsMax)
	`OVS_MAX_x1: wDGLIM = ((wMCKLIM>>`OVS_MAX_x1)-1) * DGHW;
	`OVS_MAX_x2: wDGLIM = ((wMCKLIM>>`OVS_MAX_x2)-1) * DGHW;
	`OVS_MAX_x4: wDGLIM = ((wMCKLIM>>`OVS_MAX_x4)-1) * DGHW;
	`OVS_MAX_x8: wDGLIM = ((wMCKLIM>>`OVS_MAX_x8)-1) * DGHW;
	`OVS_MAX_x16: wDGLIM = ((wMCKLIM>>`OVS_MAX_x16)-1) * DGHW;
	`OVS_MAX_x32: wDGLIM = ((wMCKLIM>>`OVS_MAX_x32)-1) * DGHW;
	default: wDGLIM = 'hx;
	endcase
end

//*****************************************
always@ (posedge iCLK or negedge iCLRn) begin
	if (!iCLRn) begin
		rCNT <= 0;
		rBCK <= 0; rWCK <= 0;
		rSHL <= 0; rSHR <= 0;
	end
	else begin
		// Control timer to read address generate
		rCNT <= (rCNT == wCntLim) ? 0 : (rCNT+1);		
		
		// Word Clock Processing Logic
		if (rCNT == wCntLim) rWCK <= 0;
		else if (rCNT == (wCntLim - WCKW)) rWCK <= 1;
		
		// Load input data and shift to output
		if (wLOAD) begin
			rSHL <= (wDL >> (24-DLEN)) ^ wCoding;
			rSHR <= (wDR >> (24-DLEN)) ^ wCoding;
		end
		else begin
			if (BCKDIV) begin
				if (rCNT[BCKDIV-1:0] == ((1<<BCKDIV)-1)) begin
					rSHL <= rSHL << 1;
					rSHR <= rSHR << 1;
				end
			end
			else begin
				rSHL <= rSHL << 1;
				rSHR <= rSHR << 1;
			end
		end
	end
end

//**** Deglitcher signal processing ****************
always@ (posedge iCLK or negedge wDGCLR) begin
	if (!wDGCLR) rDG <= 0;
	else if (rCNT == wCntLim-1) rDG <= 1;
	else if (rCNT == ((wDGLIM>>5)-1)) rDG <= 0;
end

endmodule



















