module FIR_RAM
#( parameter W = 27 )
(
/***** Global clock signals *****/
input iCLK, iCLRn, 

/**** Control signals to FIR_RAM ****/
// read signal from DATA_READ
input iOUT_RD,
// adr read 
input[8:0] iADR_OUT, iADR_RD,
// adr write 
input[8:0] iADR_WR,
// write signal
input iWR,
/************************************/

/**** Signals from ADR_GEN **********/
input iSTART_LOAD, iSUM_LOAD, iDMUX,
input[2:0] iSTG_NUM,
/************************************/

/**** Signals to MAC ****************/
output oMAC_LOAD, oMAC_SUM, oMAC_MUX,
output[2:0] oMAC_STG,
output oMAC_ENA,
/************************************/

/**** Data Signals ******************/
// data input from MAC
input[W-1:0] DL, DR,
// data output to DATA_READ
output reg[W-1:0] QL, QR
/************************************/
);


/************** MAC Control Logic **********************/
reg[1:0] rStartLoad, rSumLoad, rDmux, rOutRead;
reg[2:0] rStgNum0, rStgNum1;

/************** FIR FIFO RAM ***************************/
// Read address logic
wire[8:0] wADR_RD = (iOUT_RD) ? iADR_OUT : iADR_RD;
// data RAM array
reg[2*W-1:0] rQ, rRAM[0:511];

/************** Output Assigments **********************/
assign oMAC_LOAD = rStartLoad[1];
assign oMAC_SUM = rSumLoad[1];
assign oMAC_MUX = rDmux[1];
assign oMAC_STG = rStgNum1;
assign oMAC_ENA = !rOutRead[1];
/*******************************************************/
always@ (posedge iCLK or negedge iCLRn) begin
	if (!iCLRn) begin
		rStartLoad <= 0; rSumLoad <= 0; rDmux <= 0; rOutRead <= 0;
		rStgNum0 <= 0; rStgNum1 <= 0;
	end	
	else begin		
		rStartLoad <= (rStartLoad << 1) | iSTART_LOAD;
		rSumLoad <= (rSumLoad << 1) | iSUM_LOAD;
		rDmux <= (rDmux << 1) | iDMUX;
		rStgNum0 <= iSTG_NUM;
		rStgNum1 <= rStgNum0;
		rOutRead <= (rOutRead << 1) | iOUT_RD;
	end
end
/*******************************************************/

/************** FIR FIFO RAM ***************************/
always@ (posedge iCLK or negedge iCLRn)
	{QL, QR} <= (!iCLRn) ? 0 : rQ;

always@ (posedge iCLK) begin
	if (iWR) rRAM[iADR_WR] <= {DL, DR};
	rQ <= rRAM[wADR_RD];
end
/*******************************************************/



endmodule





module FIR_COEF_RAM
#( parameter CW = 12, AW = 11 ) // CW - Coefficients Width, AW - Address Width
(
/***** Global clock signals *****/
input iCLRn,
input iCLKR, 			// read clock
input iCLKW, 			// write clock
// control input
input[AW-1:0] iADR_RD,	// read address
input[AW-1:0] iADR_WR,	// write address
input iWR,				// write enable
// data input
input[CW-1:0] iD,
// data output
output reg[CW-1:0] oD
);

// data registers
reg[CW-1:0] rRAM[0:(1<<AW)-1], rQ;

always@ (posedge iCLKR or negedge iCLRn) begin
	oD <= (!iCLRn) ? 0 : rQ;
end

always@ (posedge iCLKR) begin
	rQ <= rRAM[iADR_RD];
end

always@ (posedge iCLKW) begin
	if (iWR) rRAM[iADR_WR] <= iD;
end


// Initialisation coefficients table. First coefficient address of every FIR stage must be multiple by 4!
initial begin
   $readmemh("DF2_COEF_TABLE.mem", rRAM, 0);
end


endmodule


