`include "defines.v"
`include "spi_data.v"
`include "spi_gen.v"
`include "rom_cnt_cont.v"
`include "rom_dg_hi.v"
`include "rom_stinit.v"


module SAI_output
(
// input global signals
input clk, clrn, mck_mode,

// input control signals
input ob_ntwc, bck_inv, dg_inv, dg_pp, dg_off, outZ, outInv,
//input[1:0] spi_hsize
input[1:0] ovs_max, spi_dsize, dgh_mode, dgl_mode,
input[2:0] d_width, d_speed, spi_mode, head_mode,

// input data
input signed[23:0] d_left, d_right,
// output control
output load,

// output data
output dat1_left, dat1_right, bck1, wck1, ltch1,
output dat2_left, dat2_right, bck2, wck2, ltch2,
output dg
);

// control registers
reg[7:0] cnt;
reg rload, rstart, rppln, rstart1, rstart2;

// control signals
reg[7:0] initcnt, rstinit;
reg[1:0] spi_hsize;
reg wdiff;	// differential mode
reg wppln;	// pipeline mode
reg wddm;	// dual data serial mode
reg wcont;	// bck continue mode
reg wck_mode;
reg rdg;
reg wspi_size, wspi_mode;
wire[7:0] wicnt_cont, wicnt, wdg_hi;

// signals interconnects
reg[7:0] whead1, whead2;
reg[7:0] whead1_left, whead2_left, whead1_right, whead2_right;
reg[2:0] wbck_num;
wire wdat1_left, wdat1_right, wbck1, wwck1, wltch1;
wire wdat2_left, wdat2_right, wbck2, wwck2, wltch2;

// spi_data interconnects
reg wsdi_left, wsdi_right;

// spi_gen interconnects
wire wshift1, wload1, wshift2, wload2;

// interconnects assigments
assign load = rload;

// output assigments
assign dg = (rdg ^ dg_inv);
assign dat1_left = (outZ) ? 1'bZ : wdat1_left;
assign dat1_right = (outZ) ? 1'bZ : wdat1_right;
assign bck1 = (outZ) ? 1'bZ : wbck1;
assign wck1 = (outZ) ? 1'bZ : wwck1;
assign ltch1 = wltch1;
assign dat2_left = (outZ) ? 1'bZ : wdat2_left;
assign dat2_right = (outZ) ? 1'bZ : wdat2_right;
assign bck2 = (outZ) ? 1'bZ : wbck2;
assign wck2 = (outZ) ? 1'bZ : wwck2;
assign ltch2 = wltch2;


// async logic
always@( 1 ) begin
	case( spi_mode )
	`SPI_DIFF_MODE:		begin wdiff = 1; wppln = 0; wddm = 0; wcont = 0; end	// diff
	`SPI_DIFF_S_MODE:	begin wdiff = 1; wppln = 0; wddm = 1; wcont = 0; end	// diff S *
	`SPI_PPLN_MODE: 	begin wdiff = 0; wppln = 1; wddm = 0; wcont = 0; end	// conv
	`SPI_PPLN_S_MODE:	begin wdiff = 1; wppln = 1; wddm = 0; wcont = 0; end	// conv S
	`SPI_CH_S_MODE:		begin wdiff = 1; wppln = 0; wddm = 1; wcont = 0; end	// ch S	*
	`SPI_RJ24_MODE:		begin wdiff = 1; wppln = 0; wddm = 0; wcont = 1; end	// RJ24
	default: 			begin wdiff = 1; wppln = 0; wddm = 0; wcont = 0; end	// diff
	endcase
	
	if( spi_dsize == `SPI_DATA_DEFAULT ) begin
		whead1 = 8'hxx; whead2 = 8'hxx; spi_hsize = `SPI_HEADER_NONE; wck_mode = 0;
	end
	else begin
		case( head_mode )
/* 0 */	`SPI_HEAD_FREE:		begin whead1 = `HEAD1_NONE;		whead2 = `HEAD2_NONE;		spi_hsize = `SPI_HEADER_NONE; end
/* 1 */	`SPI_HEAD_DAC8812:	begin whead1 = `HEAD1_DAC8812;	whead2 = `HEAD2_DAC8812;	spi_hsize = `SPI_HEADER_2bit; end
/* 2 */	`SPI_HEAD_AD5791:	begin whead1 = `HEAD1_AD5791;	whead2 = `HEAD2_AD5791;		spi_hsize = `SPI_HEADER_4bit; end
/* 3 */	`SPI_HEAD_AD5063:	begin whead1 = `HEAD1_AD5063;	whead2 = `HEAD2_AD5063;		spi_hsize = `SPI_HEADER_8bit; end
/* 4 */	`SPI_HEAD_DAC11001: begin whead1 = `HEAD1_DAC11001;	whead2 = `HEAD2_DAC11001;	spi_hsize = `SPI_HEADER_8bit; end
/* 5 */	`SPI_HEAD_LTC1592:	begin whead1 = `HEAD1_LTC1592;	whead2 = `HEAD2_LTC1592;	spi_hsize = `SPI_HEADER_8bit; end
/* 6 */	`SPI_HEAD_LTC2752:	begin whead1 = `HEAD1_LTC2752;	whead2 = `HEAD2_LTC2752;	spi_hsize = `SPI_HEADER_8bit; end
/* 7 */	`SPI_HEAD_RESERVE:	begin whead1 = `HEAD1_NONE;		whead2 = `HEAD2_NONE;		spi_hsize = `SPI_HEADER_NONE; end
		endcase
		
		wck_mode = 1;
	end
	
	if( spi_mode == `SPI_PPLN_S_MODE ) begin	// pipeline serial
		whead1_left = (rppln) ? whead1 : whead2;
		whead2_left = (rppln) ? whead1 : whead2;
		whead1_right = (rppln) ? whead1 : whead2;
		whead2_right = (rppln) ? whead1 : whead2;
	end
	else if( spi_mode == `SPI_CH_S_MODE ) begin 	// channels serial	*
		whead1_left = whead1;
		whead2_left = whead1;
		whead1_right = whead2;
		whead2_right = whead2;
	end
	else begin // diff, diff serial, pipeline
		whead1_left = whead1;
		whead2_left = whead2;
		whead1_right = whead1;
		whead2_right = whead2;
	end
	
	wsdi_left = (spi_mode == `SPI_CH_S_MODE) ? dat1_right : dat2_left;
	wsdi_right = (spi_mode == `SPI_CH_S_MODE) ? dat1_left : dat2_right;
	
	if( spi_mode == `SPI_RJ24_MODE )
		wbck_num = `BCK_NUM_CON;	// continues mode
	else if( spi_hsize == `SPI_HEADER_NONE ) // no header
		case( spi_dsize )
		`SPI_DATA_DEFAULT: wbck_num = d_width;		// data width
		`SPI_DATA_16bit: wbck_num = `BCK_NUM_16;	// 16 bit
		`SPI_DATA_20bit: wbck_num = `BCK_NUM_20;	// 20 bit
		`SPI_DATA_24bit: wbck_num = `BCK_NUM_24;	// 24 bit
		endcase
	else if( spi_hsize == `SPI_HEADER_2bit ) // 2 bit header
		case( spi_dsize )	
		`SPI_DATA_DEFAULT: wbck_num = d_width;		// data width
		`SPI_DATA_16bit: wbck_num = `BCK_NUM_18;	// 16+2 bit
		`SPI_DATA_20bit: wbck_num = `BCK_NUM_22;	// 20+2 bit
		`SPI_DATA_24bit: wbck_num = `BCK_NUM_32;	// 24+2 bit
		endcase
	else if( spi_hsize == `SPI_HEADER_4bit ) // 4 bit header
		case( spi_dsize )	
		`SPI_DATA_DEFAULT: wbck_num = d_width;		// data width
		`SPI_DATA_16bit: wbck_num = `BCK_NUM_20;	// 16+4 bit
		`SPI_DATA_20bit: wbck_num = `BCK_NUM_24;	// 20+4 bit
		`SPI_DATA_24bit: wbck_num = `BCK_NUM_32;	// 24+4 bit
		endcase
	else if( spi_hsize == `SPI_HEADER_8bit ) // 8 bit header
		case( spi_dsize )	
		`SPI_DATA_DEFAULT: wbck_num = d_width;		// data width
		`SPI_DATA_16bit: wbck_num = `BCK_NUM_24;	// 16+8 bit
		`SPI_DATA_20bit: wbck_num = `BCK_NUM_32;	// 20+8 bit
		`SPI_DATA_24bit: wbck_num = `BCK_NUM_32;	// 24+8 bit
		endcase
		
	wspi_size = (spi_dsize == `SPI_DATA_DEFAULT) ? 1 : 0;
end

always@( posedge clk or negedge clrn ) begin
	if( !clrn ) begin
		cnt <= 0; rload <= 0; rstart <= 0; rstart1 <= 0; rstart2 <= 0;
		rdg <= 0; initcnt <= 0; rstinit <= 0;
	end
	else begin
		initcnt <= (mck_mode) ? (191 >> ovs_max) : (255 >> ovs_max);
		cnt <= (cnt < initcnt) ? (cnt + 1) : 0;
		
		rstinit <= (spi_mode == `SPI_RJ24_MODE) ? wicnt_cont : (initcnt - wicnt);
		rstart <= (cnt == rstinit) ? 1 : 0;
		
		if( wppln && spi_mode == 2 ) begin
			rstart1 <= (rppln) ? 0 : rstart;
			rstart2 <= (rppln) ? rstart : 0;
		end
		else begin
			rstart1 <= rstart;
			rstart2 <= rstart;
		end
		
		rload <= wload1 | wload2;

		if( dg_off ) rdg <= 0;
		else if( dg_pp ) rdg <= (cnt == 0) ? (~rdg) : rdg;
		else rdg <= (cnt == wdg_hi) ? 0 : ( (cnt == 0) ? 1 : rdg );
	end
end

always@( posedge clk or negedge wppln )
	rppln <= (!wppln) ? 0 : ( (rstart) ? (~rppln) : rppln );

rom_cnt_cont __rom_cnt_cont(
	// input control
	.adr( {ovs_max[1:0], mck_mode, d_speed[2:0]} ),

	// output data
	.q( wicnt_cont )
);

rom_dg_hi __rom_dg_hi(
	// input control
	.adr( {ovs_max[1:0], mck_mode, dgh_mode[1:0]} ),

	// output data
	.q( wdg_hi )
);

rom_stinit __rom_stinit(
	// input control
	.adr( {wspi_size, dgl_mode[1:0], d_speed[2:0]} ),

	// output data
	.q( wicnt )
);
	
spi_data spi_data_left(
	// input global signals
	.clk( clk ), .clrn( clrn ), 
	// input control signals
	.ob_ntwc( ob_ntwc ), .diff( wdiff ), .ddm( wddm ), .outInv( outInv ),
	.load1( wload1 ), .load2( wload2 ), .ena1( wshift1 ), .ena2( wshift2 ),
	.sd_mode( spi_dsize ), .head_size( spi_hsize ),
	// serial input
	.sd( wsdi_left ),
	// input data
	.d( d_left ),
	.head1( whead1_left ), .head2( whead2_left ),

	// output data
	.sq1( wdat1_left ), .sq2( wdat2_left )
);

spi_data spi_data_right(
	// input global signals
	.clk( clk ), .clrn( clrn ), 
	// input control signals
	.ob_ntwc( ob_ntwc ), .diff( wdiff ), .ddm( wddm ), .outInv( outInv ),
	.load1( wload1 ), .load2( wload2 ), .ena1( wshift1 ), .ena2( wshift2 ),
	.sd_mode( spi_dsize ), .head_size( spi_hsize ),
	// serial input
	.sd( wsdi_right ),
	// input data 
	.d( d_right ), 
	.head1( whead1_right ), .head2( whead2_right ),

	// output data
	.sq1( wdat1_right ), .sq2( wdat2_right )
);

spi_gen spi_gen1(
	// input global signals
	.clk( clk ), .clrn( clrn ),
	// input control signals
	.start( rstart1 ), .bck_inv( bck_inv ), .ddm( wddm ), .cont( wcont ), .outInv( outInv ),
	.bck_num( wbck_num ), .bck_mode( d_speed ), .wck_mode( wck_mode ),

	// output control
	.shift( wshift1 ), .load( wload1 ),
	// output data
	.bck( wbck1 ), .wck( wwck1 ), .lat( wltch1 )
);

spi_gen spi_gen2(
	// input global signals
	.clk( clk ), .clrn( clrn ),
	// input control signals
	.start( rstart2 ), .bck_inv( bck_inv ), .ddm( wddm ), .cont( wcont ), .outInv( outInv ),
	.bck_num( wbck_num ), .bck_mode( d_speed ), .wck_mode( wck_mode ),

	// output control
	.shift( wshift2 ), .load( wload2 ),
	// output data
	.bck( wbck2 ), .wck( wwck2 ), .lat( wltch2 )
);




endmodule





















