
#include "../../app/system.h"
#include "usb_Descriptors.h"
#include "usb_PL.h"




/*****************************************************************************
 Data Types
 ****************************************************************************/
typedef struct{ // feedback freq values in 16.16 format
    uint32_t nom;   // nominal frequency value
    uint32_t lo;    // low limit frequency
    uint32_t hi;    // high limit frequency
}tFbSettFreq;

typedef struct{ // Data size to Input Stream transmitt
    uint16_t nom;   // nominal size value
    uint16_t lo;    // low size value
    uint16_t hi;    // high size value
}tBlockSize;


/*****************************************************************************
 Constatnt data
 ****************************************************************************/
/* 
 * tFeedBackFreq - Frequences array to feedback with +/-400ppm limits 
 */
static const tFbSettFreq tFeedBackFreq[USB_FREQ_NUM] = { 
    {
     _FB_FREQ_VAL_GET(PULL_FREQ_NOM_44k, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_LOW_44k, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_HI_44k, EP1RX_PULL_TIME_mS)
 }, {
     _FB_FREQ_VAL_GET(PULL_FREQ_NOM_48k, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_LOW_48k, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_HI_48k, EP1RX_PULL_TIME_mS)
 }, {
     _FB_FREQ_VAL_GET(PULL_FREQ_NOM_44k * 2, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_LOW_44k * 2, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_HI_44k * 2, EP1RX_PULL_TIME_mS)
 }, {
     _FB_FREQ_VAL_GET(PULL_FREQ_NOM_48k * 2, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_LOW_48k * 2, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_HI_48k * 2, EP1RX_PULL_TIME_mS)
 }, {
     _FB_FREQ_VAL_GET(PULL_FREQ_NOM_44k * 4, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_LOW_44k * 4, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_HI_44k * 4, EP1RX_PULL_TIME_mS)
 }, {
     _FB_FREQ_VAL_GET(PULL_FREQ_NOM_48k * 4, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_LOW_48k * 4, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_HI_48k * 4, EP1RX_PULL_TIME_mS)
 }, {
     _FB_FREQ_VAL_GET(PULL_FREQ_NOM_44k * 8, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_LOW_44k * 8, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_HI_44k * 8, EP1RX_PULL_TIME_mS)
 }, {
     _FB_FREQ_VAL_GET(PULL_FREQ_NOM_48k * 8, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_LOW_48k * 8, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_HI_48k * 8, EP1RX_PULL_TIME_mS)
 }, {
     _FB_FREQ_VAL_GET(PULL_FREQ_NOM_44k * 16, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_LOW_44k * 16, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_HI_44k * 16, EP1RX_PULL_TIME_mS)
 }, {
     _FB_FREQ_VAL_GET(PULL_FREQ_NOM_48k * 16, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_LOW_48k * 16, EP1RX_PULL_TIME_mS),
     _FB_FREQ_VAL_GET(PULL_FREQ_HI_48k * 16, EP1RX_PULL_TIME_mS)
 } };
/* 
 * Data size array to EP5 transmitt
 */
static const tBlockSize tEp5DataSize[USB_FREQ_NUM] = {
    {48, 40, 56},
    {48, 40, 56},
    {88, 80, 96},
    {96, 88, 104},
    {176, 168, 184},
    {192, 184, 200},
    {352, 344, 360},
    {384, 376, 392},
    {704, 696, 712},
    {768, 760, 776}
};


/*****************************************************************************
 Local variables
 ****************************************************************************/
static tUSB_CON_STAT tUsbControlStatus = {0};

// EP0 buffer and control variables
static uint8_t ep0buf[64] __attribute__((aligned (4)));
static uint8_t statusAnswer[] = {0x00, 0x00}; // Double Zero

// EP5_IN buffer and control variables
static uint8_t ep5buf[INPUT_TRANSFER_FIFO_SIZE] __attribute__((coherent)) __attribute__((aligned (16)));
static uint16_t ep5txsize;  // size of data to USB transmitt
static tBlockSize tEp5BlockSize; // size of data to transimt to host in one packet
static tFIFO_CONTROL tEp5Fifo;

/*****************************************************************************
 Global variables
 ****************************************************************************/



/*
    usb_PL_init()
 */
void usb_PL_init (void)
{
    // EP-IN data buffer initialisation
    tEp5Fifo.adr = KVA_TO_PA(ep5buf);
    // EP-FeedBack variables initialisation
    tEp5BlockSize = tEp5DataSize[USB_SFREQ_44k];

    tUsbControlStatus.streamSamplingFreq[eSTREAM_INPUT] = USB_SFREQ_44k;

    _DMA_InputDestinationAddress_Set(tEp5Fifo.adr);
    
    usb_LL_Init();
}


/*
    usb_PL_InterfaceState_Update()
 */
static inline void usb_PL_InterfaceState_Update ( USB_SETUP_PACKET *p )
{
    tUsbControlStatus.streamInterfaceAltsetting[eSTREAM_INPUT] = p->bAltID;
    
    if (p->bAltID == USB_AUDIO_INPUT_ALTSET_2CH_ON) {
        // Set sampling frequency settings
        tEp5BlockSize = tEp5DataSize[tUsbControlStatus.streamSamplingFreq[eSTREAM_INPUT]];
        ep5txsize = tEp5BlockSize.nom;
        // Calculate buffer length in dependence of sampling frequency
        tEp5Fifo.len = sizeof(ep5buf) >> ( (USB_SFREQ_768k - tUsbControlStatus.streamSamplingFreq[eSTREAM_INPUT]) >> 1 );
        tEp5Fifo.min = tEp5Fifo.len >> 4;
        tEp5Fifo.max = tEp5Fifo.len >> 3;
        tEp5Fifo.ptr = tEp5Fifo.len >> 1;
        tEp5Fifo.rem = 0;
        _DMA_Input_Transfer_Size_Set(tEp5Fifo.len);
        _PORT_Input_Transfer_Sync();    // Start Input Transfer on edge LRCK input
        _USBDMA_EP5TX_Transfer_Start(tEp5Fifo.ptr, ep5txsize);
        tEp5Fifo.ptr += ep5txsize;
        PIC_DATAFLOW_ON();
    }
    else if (p->bAltID == USB_AUDIO_INPUT_ALTSET_OFF) {
        _I2S_Input_Transfer_Stop();
        _DMA_Input_Transfer_Stop();
        PIC_DATAFLOW_OFF();
    }
}

/*
    usb_PL_SamplingFreq_Set()
 */
static inline void usb_PL_SamplingFreq_Set ( void )
{
    uint32_t freq;
    usb_LL_Ep0_BulkRead( &freq, sizeof(freq) );

    eUSB_SAMPLING_FREQ sfreq;

    switch (freq) {
        case 44100: sfreq = USB_SFREQ_44k; break;
        case 48000: sfreq = USB_SFREQ_48k; break;
        case 88200: sfreq = USB_SFREQ_88k; break;
        case 96000: sfreq = USB_SFREQ_96k; break;
        case 176400: sfreq = USB_SFREQ_176k; break;
        case 192000: sfreq = USB_SFREQ_192k; break;
        case 352800: sfreq = USB_SFREQ_352k; break;
        case 384000: sfreq = USB_SFREQ_384k; break;
        case 705600: sfreq = USB_SFREQ_705k; break;
        case 768000: sfreq = USB_SFREQ_768k; break;
        default: return;
    }

    tUsbControlStatus.streamSamplingFreq[eSTREAM_INPUT] = sfreq;
    
    if (tUsbControlStatus.streamInterfaceAltsetting[eSTREAM_INPUT] == USB_AUDIO_INPUT_ALTSET_2CH_ON) {
        _I2S_Input_Transfer_Stop();
    }
    
    switch (tUsbControlStatus.streamSamplingFreq[eSTREAM_INPUT]) {
        default: // 44/48k
            FS0_SET_LOW();
            FS1_SET_LOW();
            break;

        case USB_SFREQ_88k:
        case USB_SFREQ_96k:
            FS0_SET_HIGH();
            FS1_SET_LOW();
            break;

        case USB_SFREQ_176k:
        case USB_SFREQ_192k:
            FS0_SET_LOW();
            FS1_SET_HIGH();
            break;
            
        case USB_SFREQ_352k:
        case USB_SFREQ_384k:
        case USB_SFREQ_705k:
        case USB_SFREQ_768k:
            break;
    }

    if (tUsbControlStatus.streamSamplingFreq[eSTREAM_INPUT] & 0x1)
    {
        OSC_44K_OFF();
        OSC_48K_ON();
    }  // Set 48kHz clock domain
    else
    {
        OSC_48K_OFF();
        OSC_44K_ON(); // Set 44.1kHz clock domain
    }
    
    if (tUsbControlStatus.streamInterfaceAltsetting[eSTREAM_INPUT] == USB_AUDIO_INPUT_ALTSET_2CH_ON) {
        _PORT_Input_Transfer_Sync();    // Start Input Transfer on edge LRCK input
    }
}

/*
    usb_PL_GetDescriptor()
 */
static inline int usb_PL_GetDescriptor ( USB_SETUP_PACKET *p, const void **dp )
{
    int len = 0;
    
    switch (p->bDescriptorType)
    {
        case USB_DESCRIPTOR_DEVICE:
            *dp = tUsbDescriptors.device.p;
            len = tUsbDescriptors.device.size;
            break;
            
        case USB_DESCRIPTOR_CONFIGURATION:
            *dp = tUsbDescriptors.configuration.p;
            len = tUsbDescriptors.configuration.size;
            break;
            
        case USB_DESCRIPTOR_STRING:
            if (p->bDscIndex < STRING_DESCRIPTORS_NUM) {
                *dp = tUsbDescriptors.string[p->bDscIndex].p;
                len = tUsbDescriptors.string[p->bDscIndex].size;
            }
            break;
            
        case USB_DESCRIPTOR_DEVICE_QUALIFIER:
            *dp = tUsbDescriptors.qualifier.p;
            len = tUsbDescriptors.qualifier.size;
            break;
    }

    return len;
}
 
//uint64_t DebugCmd[256] = {0};     // Debug array to logging control cmd
//int DebugOffset = 0;
/*
    usb_PL_ReqProcess()
 */
static inline void usb_PL_ReqProcess ( void *pbuf )
{
    USB_SETUP_PACKET *p = pbuf;
//    if (DebugOffset < (sizeof(DebugCmd)/8)) DebugCmd[DebugOffset++] = *(uint64_t*)pbuf;
    if (p->DataDir == USB_SETUP_REQUEST_DIRECTION_HOST_TO_DEVICE)
    {
        switch (p->bmbRequest)
        {
            /***************************************************/
            /************ Standart requests ********************/
            /***************************************************/
            case STDREQ_CLEARE_FEATURE_DEVICE:
            case STDREQ_CLEARE_FEATURE_INTERFACE:
            case STDREQ_CLEARE_FEATURE_ENDPOINT:
                USBE0CSR0bits.STALL = 1;
                break;

            case STDREQ_SET_CONFIGURATION:
                tUsbControlStatus.configuration = p->bConfigurationValue;
                break;

            case STDREQ_SET_FEATURE_DEVICE:
                tUsbControlStatus.setFeature[USB_SETUP_REQUEST_RECIPIENT_DEVICE] = p->bFeature;
                break;

            case STDREQ_SET_FEATURE_INTERFACE:
                tUsbControlStatus.setFeature[USB_SETUP_REQUEST_RECIPIENT_INTERFACE] = p->bFeature;
                break;

            case STDREQ_SET_FEATURE_ENDPOINT:
                tUsbControlStatus.setFeature[USB_SETUP_REQUEST_RECIPIENT_ENDPOINT] = p->bFeature;
                break;

            case STDREQ_SET_INTERFACE:
                usb_PL_InterfaceState_Update( p );
                break;

            /***************************************************/
            /********** Audio Device - specific requests *******/
            /***************************************************/
            case CSREQ_SET_CURR_INTERFACE:
                switch (p->W_Value.byte.HB)
                {
                    case USB_AUDIO_V2_CS_SAM_FREQ_CONTROL:
                    {
                        switch (p->bIntfID_H) {
                            case AUDIO_SCLOCK_TERMINAL_INPUT: usb_PL_SamplingFreq_Set(  ); break;
                        }
                        break;
                    }
                    break;
                }
                break;
                
            default:
                USBE0CSR0bits.STALL = 1;
                break;
        }
    }
    else // p->DataDir == USB_SETUP_REQUEST_DIRECTION_DEVICE_TO_HOST
    {
        switch (p->bmbRequest)
        {
            /***************************************************/
            /************ Standart requests ********************/
            /***************************************************/
            case STDREQ_GET_CONFIGURATION:
                usb_LL_Ep0_BulkWrite( &tUsbControlStatus.configuration, sizeof(tUsbControlStatus.configuration) );
                break;

            case STDREQ_GET_DESCRIPTOR:
                {
                    const void *dp = NULL;
                    int len = usb_PL_GetDescriptor( p, &dp );
                    // If the descriptor is longer than the wLength field,
                    // only the initial bytes of the descriptor are returned
                    if (len > p->wLength)
                        len = p->wLength;
                   
                    if (len) usb_LL_Ep0_BulkWrite( dp, len );
                    else USBE0CSR0bits.STALL = 1;
                }
                break;

            case STDREQ_GET_INTERFACE:
                switch (p->bIntfID) {                 
                    case AUDIO_STREAMING_INPUT_INTERFACE_NUM: 
                        usb_LL_Ep0_BulkWrite( &tUsbControlStatus.streamInterfaceAltsetting[eSTREAM_INPUT], 
                                            sizeof(tUsbControlStatus.streamInterfaceAltsetting[eSTREAM_INPUT]) );
                        break;
                }
                break;

            case STDREQ_GET_STATUS_DEVICE:
            case STDREQ_GET_STATUS_INTERFACE:
            case STDREQ_GET_STATUS_ENDPOINT:
                usb_LL_Ep0_BulkWrite( statusAnswer, sizeof(statusAnswer) );
                break;

            /***************************************************/
            /********** Audio Device - specific requests *******/
            /***************************************************/
            case CSREQ_GET_RANGE_INTERFACE: 
                switch (p->W_Value.byte.HB)
                {
                    case USB_AUDIO_SAMPLING_FREQ_CONTROL:
                    {
                        const tUSB_AUDIO_REQ_L3_PARAM_BLOCK tRangeInput __attribute__ ((aligned (4))) = 
                            {   
                                (USB_SFREQ_192k+1),
                                {
                                    {44100, 44100, 0},
                                    {48000, 48000, 0},
                                    {88200, 88200, 0},
                                    {96000, 96000, 0},
                                    {176400, 176400, 0},
                                    {192000, 192000, 0},
                                }
                            };
                        
                        const tUSB_AUDIO_REQ_L3_PARAM_BLOCK *ptRange;
                        
                        switch (p->bIntfID_H) {
                            case AUDIO_SCLOCK_TERMINAL_INPUT: ptRange = &tRangeInput; break; // Input stream
                        }
                        
                        int len = sizeof(tUSB_AUDIO_REQ_L3_PARAM_BLOCK);
                        
                        if (p->wLength < len)
                            len = p->wLength;

                        usb_LL_Ep0_BulkWrite( ptRange, len );
                    }
                    break;
                }
                break;
                
            case CSREQ_GET_CURR_INTERFACE: 
                switch (p->W_Value.byte.HB)
                {
                    case USB_AUDIO_SAMPLING_FREQ_CONTROL:
                    {
                        const uint32_t afreq[] = {44100, 48000, 88200, 96000, 176400, 192000};
                        usb_LL_Ep0_BulkWrite( &afreq[tUsbControlStatus.streamSamplingFreq[eSTREAM_INPUT]], sizeof(uint32_t) );
                    }
                    break;
                }
                break;

            default:
                USBE0CSR0bits.STALL = 1;
                break;
        }
    }
}

/*
    usb_PL_tasks()
 */
void usb_PL_tasks (void)
{
    if (_System_Task_Check( SYS_TASK_USB_CONNECT )) {
        _USB_SOFT_CONNECT();        // The USB D+/D- lines are enabled and active
        _System_Task_Clr( SYS_TASK_USB_CONNECT );
    }
    
    if (_System_Task_Check( SYS_TASK_USB_EP0_ISR )) {
        usb_PL_ReqProcess( ep0buf ); // Process request

        if (USBE0CSR0bits.SETEND)
            USBE0CSR0bits.SETENDC = 1;

        _System_Task_Clr( SYS_TASK_USB_EP0_ISR );
    }
}

/* 
 * USB_Interrupt()
 */
void __ISR(_USB_VECTOR, ipl3SRS) USB_Interrupt(void)
{
    /* 
     * Read interrupt flags registers
     * Reading from USBCSRx registers directly can reset some interrupt flags!!!
     */
    uint32_t usbcsr0 = USBCSR0;
    uint32_t usbcsr1 = USBCSR1;
    uint32_t usbcsr2 = USBCSR2;

    if (usbcsr2 & _USBCSR2_RESETIF_MASK) {
        usb_LL_initReset();
     }
    
    /* Control Endpoint 0 Requests Handler */
    if (usbcsr0 & _USBCSR0_EP0IF_MASK) {
        if (tUsbControlStatus.adr_pending == true) { // Set Address, upon first IN transaction
            USBCSR0bits.FUNC = tUsbControlStatus.adr;
            tUsbControlStatus.adr_pending = false;
        }
        
        if (_System_Task_Check(SYS_TASK_USB_EP0_ISR) == 0) // if no request processing
        if (USBE0CSR0bits.RXRDY) // if received control packet
        {   // read control cmd
            _EP0_Control_Cmd_Read( ep0buf );
            // Fast checking "set address" request (2mS processing timeout)
            if (((USB_SETUP_PACKET*)ep0buf)->bmbRequest == STDREQ_SET_ADDRESS) {
                tUsbControlStatus.adr = ((USB_SETUP_PACKET*)ep0buf)->bDevADR;
                tUsbControlStatus.adr_pending = true;
            }
            else
                _System_Task_Set( SYS_TASK_USB_EP0_ISR );   // Set delayed task
        }
        
        USBCSR0bits.EP0IF = 0;      // Clear the USB EndPoint 0 Interrupt Flag.
    }
    
    /* Endpoint 5 TX Interrupt Handler */
    if (usbcsr0 & _USBCSR0_EP5TXIF_MASK) {
        if (!USBE5CSR0bits.TXPKTRDY)
        { // ISOCHRONOUS Endpoint 5 Transmitt A Packet
            uint16_t txcnt = ep5txsize;                     // Load transmitting data count
            uint32_t adr = tEp5Fifo.adr + tEp5Fifo.ptr;     // Calculate USB DMA5 memory address            
            tEp5Fifo.ptr += txcnt;                          // Buffer pointer increment

            if (tEp5Fifo.ptr >= tEp5Fifo.len) { // if not anoth space if data buffer - transmitt part only
                tEp5Fifo.rem = tEp5Fifo.ptr - tEp5Fifo.len; // Calculate remain value
                tEp5Fifo.ptr = tEp5Fifo.rem;                // Set pointer to remain value
                txcnt -= tEp5Fifo.rem;                      // Set USB DMA5 data lenght
            }
            
            _USBDMA_EP5TX_Transfer_Start(adr, txcnt);
        }
    }

    /* EP5 Errors Processing */
    if (USBE5CSR0bits.UNDERRUN) {   // An IN token has been received when TXPKTRDY is not set
        USBE5CSR0bits.UNDERRUN = 0; // Clear the TX UNDERRUN
    }
    
    /* Interrupt Flags Processing */
    if (USBCSR2 & 0x00ff0000) {
        USBCSR2 &= ~(0x00ff0000);
    }
    
    IFS4CLR = _IFS4_USBIF_MASK;             // Clear the USB interrupt flag
}


// Debug variables to isocronous feedback check
//unsigned int fbLowCnt = 0;
//unsigned int fbHighCnt = 0;
//unsigned int fbNomCnt = 0;
/* 
 * USBDMA_Interrupt()
 */
void __ISR(_USB_DMA_VECTOR, ipl4SRS) USBDMA_Interrupt(void)
{
    uint32_t dmaInt = USBDMAINT;    // cleare flags on read
    
    if (dmaInt & _USBDMAINT_DMA5IF_MASK)
    {
        if (tEp5Fifo.rem) {
            _USBDMA_EP5TX_Transfer_Start(tEp5Fifo.adr, tEp5Fifo.rem);
            tEp5Fifo.rem = 0;
        }
        else {
            USBE5CSR0bits.TXPKTRDY = 1;                     // Confirm to send data packet
            
            static uint16_t speed_check = 0;                // checking data speed interval counter
            speed_check++;
            speed_check &= 0x3;
            if (!speed_check) { // check every 4-t transaction
                int buff_diff = DMA_INPUT_TRANSFET_PTR - tEp5Fifo.ptr; // Calculate Read and Write data difference
                if (buff_diff < 0)
                    buff_diff += tEp5Fifo.len;

                // update actual block data lenght
                if (buff_diff < tEp5Fifo.min) {         // if usb transmitting speed is too high
                    ep5txsize = tEp5BlockSize.lo;       // turn down one sample from packet lenght
                }
                else if (buff_diff > tEp5Fifo.max) {    // if usb transmitting speed is too low
                    ep5txsize = tEp5BlockSize.hi;       // add one sample to packet lenght (4byte * 2channels)
                }
                else {
                    ep5txsize = tEp5BlockSize.nom;
                }
            }
        }
    }
    
    IFS4CLR = _IFS4_USBDMAIF_MASK;              // Clear the USB DMA interrupt flag.
}


