`include "defines.v"
`include "NoiseShape.v"

module data_read
(
// input global signals
input clk, clrn,
// input control signals
input ready, load, dithen, shpen,
input[1:0] ovs_max,
input[2:0] d_width,
input[5:0] curr_pos,
// input data
input signed[25:0] d_left, d_right,

// output control
output[8:0] q_adr,
// output data
output signed[23:0] q_left, q_right,
// output signals
output clipn
);

// data registers
reg[24:0] rleft, rright;

// control registers
reg[5:0] adrcnt;
reg[1:0] dith_mode;
reg rload, rovf; 
reg dnovf_left, upovf_left, ovfok_left;
reg dnovf_right, upovf_right, ovfok_right;

// control signals
reg[25:0] sum_left, sum_right, round_left, round_right;
reg[5:0] cntsync;
reg[3:0] trunc;
reg sample;
reg wdithen;
wire signed[24:0] dither;

// output data assigments
assign q_left = rleft;
assign q_right = rright;
assign q_adr = {3'h6, adrcnt[5:0]};
assign clipn = rovf;

// async logic overflow processing
always@( 1 ) begin
	cntsync = (adrcnt == curr_pos) ? ((-4) << ovs_max) : 1;
	sample = rload & ready;
		
	// left channel overflow processing
	dnovf_left = (rleft[24] ^ rleft[23]) ? rleft[24] : 0;
	upovf_left = (rleft[24] ^ rleft[23]) ? (~rleft[24]) : 0;
	ovfok_left = (rleft[24] ^ rleft[23]) ? 0 : 1;

	// right channel overflow processing
	dnovf_right = (rright[24] ^ rright[23]) ? rright[24] : 0;
	upovf_right = (rright[24] ^ rright[23]) ? (~rright[24]) : 0;
	ovfok_right = (rright[24] ^ rright[23]) ? 0 : 1;
	
		`ifdef LOGIC_STYLE_LATTICE
	if( upovf_left ) round_left = 26'h0ffffff;
	else if( dnovf_left ) round_left = 26'h3000000;
	else round_left = dither;
	
	if( upovf_right ) round_right = 26'h0ffffff;
	else if( dnovf_right ) round_right = 26'h3000000;
	else round_right = dither;
	
	sum_left = d_left * ovfok_left + round_left;
	sum_right = d_right * ovfok_right + round_right;
		`endif
	
		`ifdef LOGIC_STYLE_ALTERA
	if( upovf_left ) sum_left = 26'h0ffffff;
	else if( dnovf_left ) sum_left = 26'h3000000;
	else sum_left = d_left + dither;
	
	if( upovf_right ) sum_right = 26'h0ffffff;
	else if( dnovf_right ) sum_right = 26'h3000000;
	else sum_right = d_right + dither;
		`endif
	
	if( !clrn )
		trunc = 4'h0;
	else
		case( d_width )
		`DATA_16bit: trunc = 4'h0;
		`DATA_18bit: trunc = 4'h1;
		`DATA_20bit: trunc = 4'h3;
		`DATA_22bit: trunc = 4'h7;
		`DATA_24bit: trunc = 4'hf;
		default: trunc = 4'hf;
		endcase
end

always@( posedge clk or negedge clrn ) begin
	if( !clrn ) begin
		adrcnt <= 0;
		rload <= 0; rleft[24:8] <= 0; rright[24:8] <= 0;
		rovf <= 0;
	end
	else begin
		rload <= (load) ? 1 : ((ready) ? 0 : rload);
		
		if( sample ) adrcnt <= adrcnt + cntsync;
		// 16bit data
		if( sample | upovf_left | dnovf_left ) rleft[24:8] <= sum_left[25:9];
		if( sample | upovf_right | dnovf_right ) rright[24:8] <= sum_right[25:9];
		
		rovf <= ovfok_left & ovfok_right;
	end
end

always@( posedge clk or negedge trunc[0] ) begin
	if( !trunc[0] ) begin
		rleft[7:6] <= 0; rright[7:6] <= 0;
	end
	else begin // lsb 18bit data
		if( sample | upovf_left | dnovf_left ) rleft[7:6] <= sum_left[8:7];
		if( sample | upovf_right | dnovf_right ) rright[7:6] <= sum_right[8:7];
	end
end

always@( posedge clk or negedge trunc[1] ) begin
	if( !trunc[1] ) begin
		rleft[5:4] <= 0; rright[5:4] <= 0;
	end
	else begin // lsb 20bit data
		if( sample | upovf_left | dnovf_left ) rleft[5:4] <= sum_left[6:5];
		if( sample | upovf_right | dnovf_right ) rright[5:4] <= sum_right[6:5];
	end
end

always@( posedge clk or negedge trunc[2] ) begin
	if( !trunc[2] ) begin
		rleft[3:2] <= 0; rright[3:2] <= 0;
	end
	else begin // lsb 22bit data
		if( sample | upovf_left | dnovf_left ) rleft[3:2] <= sum_left[4:3];
		if( sample | upovf_right | dnovf_right ) rright[3:2] <= sum_right[4:3];
	end
end

always@( posedge clk or negedge trunc[3] ) begin
	if( !trunc[3] ) begin
		rleft[1:0] <= 0; rright[1:0] <= 0;
	end
	else begin // lsb 24bit data
		if( sample | upovf_left | dnovf_left ) rleft[1:0] <= sum_left[2:1];
		if( sample | upovf_right | dnovf_right ) rright[1:0] <= sum_right[2:1];
	end
end

NoiseShape __NShape(
	// input global sync signals
	.clk( clk ), .clrn( clrn ),
	// input control signals
	.next( sample ), .dithen( dithen ), .shpen( shpen ),
	.d_len( d_width ),
	// parallel data output
	.q( dither )
);



endmodule






















