`include "defines.v"
`include "clk_control.v"
`include "coef_control.v"
`include "init_adr.v"
`include "data_adr.v"
`include "coef_adr.v"
`include "data_write.v"
`include "data_read.v"
`include "data_ram.v"
`include "mac_control.v"

	`ifdef COEF_ATT_ENA
`include "coef_rom_9x4096.v"
`define COEF_ROM_MODULE		coef_rom_9x4096
	`else
`include "coef_rom_9x1024.v"
`define COEF_ROM_MODULE		coef_rom_9x1024
	`endif


module DF1_FIR_CORE
(
// global clock/reset inputs
input clk, clrn, mck_mode, mck_div,
// setting inputs
input sai_lrc, lp_mode, dithen, shpen,
input[1:0] ovs_max, att,
input[2:0] d_width,
// control inputs
input in_ready, out_load,
// input data
input signed[`BUS_WIDTH-1:0] d_left, d_right,

// output data
output signed[23:0] q_left, q_right,

output[2:0] fs_in,
output mcko, clipn, bypass
);

// global fir syncronisation counter
reg[1:0] syncnt;

always@( posedge clk or negedge clrn ) begin
	syncnt <= (!clrn) ? 0 : ((syncnt == 2) ? 0 : (syncnt + 1));
end

// ----------------------- midmodules wire interconnects ---------------------------------
// clk_control output signals
wire[2:0] wst_last, wfs_in;
wire wbypass;

// coef_control output signals
wire[6:0] wst1_len1, wst2_len1, wst3_len1, wst4_len1, wst5_len1;
wire[6:0] wst1_len2, wst2_len2, wst3_len2, wst4_len2, wst5_len2;
wire[9:0] wst1_adr, wst2_adr, wst3_adr, wst4_adr, wst5_adr;

// init_adr output signals
wire[6:0] init_curr, init_repnum, init_lenth1, init_lenth2;
wire[2:0] init_stage;

// data_adr output signals
wire adr_out_next, adr_out_coef_init, adr_out_data_read, adr_out_data_sync;
wire[2:0] adr_out_stg, adr_out_coef_stg;
wire[8:0] read_adr;

// coef_adr_output_signals
wire[9:0] coef_adr;

// data_write output signals
wire[6:0] curr_pos1, curr_pos2, curr_pos3, curr_pos4, curr_pos5, curr_pos6;
wire[8:0] write_adr;
wire write_enable, write_mux_in, write_fir_start;

// data_out output signals
wire[8:0] out_adr;

// RAM output signals
wire signed[`BUS_WIDTH-1:0] ram_out_left, ram_out_right;
wire ram_out_read, ram_out_start;

// coef ROM output data
wire[8:0] coef_data;

// MAC output signals
wire signed[`BUS_WIDTH-1:0] mac_out_left, mac_out_right;
wire[2:0] mac_out_stg;
// ---------------------------------------------------------------------------------------


// output assigments
assign fs_in = wfs_in;
assign bypass = wbypass;

clk_control __clk_control(
	// input global sync signals
	.clk( clk ), .clrn( clrn ), .mck_mode( mck_mode ), .mck_div( mck_div ),
	// input config
	.ovs_max( ovs_max ),
	.sai_lrc( sai_lrc ),
	// output config
	.st_last( wst_last ), .ofs_in( wfs_in ),
	.mcko( mcko ), .bypass( wbypass )
);

coef_control __coef_control(
	// input global sync signals
	.clk( clk ), .clrn( clrn ),
	// input config
	.mck_mode( mck_mode ), .lp_mode( lp_mode ),
	.ovs_max( ovs_max ),
	.fs_in( wfs_in ),
	// output coef settings
	.st1_len1( wst1_len1 ), .st2_len1( wst2_len1 ), .st3_len1( wst3_len1 ), .st4_len1( wst4_len1 ), .st5_len1( wst5_len1 ),
	.st1_len2( wst1_len2 ), .st2_len2( wst2_len2 ), .st3_len2( wst3_len2 ), .st4_len2( wst4_len2 ), .st5_len2( wst5_len2 ),
	// output coef adr
	.st1_adr( wst1_adr ), .st2_adr( wst2_adr ), .st3_adr( wst3_adr ), .st4_adr( wst4_adr ), .st5_adr( wst5_adr )
);

init_adr __init_adr( 
	// input global sync signals
	.clk( clk ), .clrn( clrn ),
	.sync( syncnt ),
	// initialisation signals
	.start( write_fir_start ), .next( adr_out_next ),
	// number of used stages
	.st_last( wst_last ),
	// input muxed data
	.curr_st1( curr_pos1 ), .curr_st2( curr_pos2 ), .curr_st3( curr_pos3 ), .curr_st4( curr_pos4 ), .curr_st5( curr_pos5 ),
	.len1_st1( wst1_len1 ), .len1_st2( wst2_len1 ), .len1_st3( wst3_len1 ), .len1_st4( wst4_len1 ), .len1_st5( wst5_len1 ),
	.len2_st1( wst1_len2 ), .len2_st2( wst2_len2 ), .len2_st3( wst3_len2 ), .len2_st4( wst4_len2 ), .len2_st5( wst5_len2 ),
	// output initialisation data
	.curr( init_curr ), .repeatnum( init_repnum ), .lenth1( init_lenth1 ), .lenth2( init_lenth2 ),
	.stage( init_stage )
);

data_adr __data_adr(
	// input global signals
	.clk( clk ), .clrn( clrn ), .bypass( wbypass ),
	.sync( syncnt ),
	// input initialisation parameters
	.curr_pos( init_curr ), .num_repeat( init_repnum ), .length_row1( init_lenth1 ), .length_row2( init_lenth2 ),
	.curr_stage( init_stage ),
	// output control siggnals
	.next_stage( adr_out_next ),
	.data_stage( adr_out_stg ),	// - flag end of MAC cicle
	.data_read( adr_out_data_read ),
	.data_sync( adr_out_data_sync ),
	// output coefficient generator control signals
	.coef_stage( adr_out_coef_stg ),
	.coef_init( adr_out_coef_init ),
	// output RAM data adress
	.data_adr( read_adr )
);

coef_adr __coef_adr(
	// input global sync signals
	.clk( clk ), .clrn( clrn ), .init( adr_out_coef_init ), .dsync( adr_out_data_sync ),
	// input initialisation parameters
	.stage_num( adr_out_coef_stg ),
	// input adresses to varios stages
	.adr_st1( wst1_adr ), .adr_st2( wst2_adr ), .adr_st3( wst3_adr ), .adr_st4( wst4_adr ), .adr_st5( wst5_adr ),
	// output ROM coefficients adress
	.coef_adr( coef_adr )
);

data_write __data_write(
	// input global sync signals
	.clk( clk ), .clrn( clrn ),
	.sync( syncnt ),
	// external signals
	.in_ready( in_ready ),
	.st_mac( mac_out_stg ),
	// output stages adress
	.currpos_st1( curr_pos1 ), .currpos_st2( curr_pos2 ), .currpos_st3( curr_pos3 ), .currpos_st4( curr_pos4 ), .currpos_st5( curr_pos5 ), .currpos_st6( curr_pos6 ),
	// write signal to RAM logic
	.write_adr( write_adr ),
	.write( write_enable ), .mux_in( write_mux_in ), .start_mac( write_fir_start )
);

data_read __data_read(
	// input global signals
	.clk( clk ), .clrn( clrn ),
	// input control signals
	.ready( ~ram_out_read ), .load( out_load ), .dithen( dithen ), .shpen( shpen ),
	.ovs_max( ovs_max ),
	.d_width( d_width ),
	.curr_pos( curr_pos6[5:0] ),
	// input data
	.d_left( (ram_out_left >> (`BUS_WIDTH - 26)) ), .d_right( (ram_out_right >> (`BUS_WIDTH - 26)) ),

	// output control
	.q_adr( out_adr ),
	// output data
	.q_left( q_left ), .q_right( q_right ),
	// output signals
	.clipn( clipn )
);

data_ram #( .W( `BUS_WIDTH ) )
__data_ram(
	// clock ipnuts
	.clk( clk ), .clrn( clrn ), 
	.iread( adr_out_data_read ), .istart( adr_out_data_sync ),
	// control adress RAM
	.inmux( write_mux_in ), .write( write_enable ),
	// input adress RAM
	.radr( read_adr ), .oadr( out_adr ),
	.wadr( write_adr ),
	// input data
	.dleft( d_left ), .dright( d_right ), // 1bit shift to overflow processing
	// mac data
	.mleft( mac_out_left ), .mright( mac_out_right ),
	// output RAM
	.qleft( ram_out_left ), .qright( ram_out_right ),
	.qread( ram_out_read ), .qstart( ram_out_start )
);

`COEF_ROM_MODULE __coef_rom(
	// global input clock
	.clk( clk ), .clrn( clrn ),
	// control input
	.adr( {att[1:0], coef_adr[9:0]} ),
	// data output
	.q( coef_data )
);

mac_control __mac_control(
	// input global signals
	.clk( clk ), .clrn( clrn ),
	.sync( syncnt ),
	// input sync signals
	.st_read( adr_out_stg ), .st_last( wst_last ),
	.read( ram_out_read ), .start( ram_out_start ),
	// input data
	.d_left( ram_out_left ), .d_right( ram_out_right ),
	// input coef
	.coef( coef_data ),
	// output data
	.q_left( mac_out_left ), .q_right( mac_out_right ),
	// output global signals
	.st_write( mac_out_stg )
);

endmodule





