/**
  ******************************************************************************
  * @file    usbd_audio_core.c
  * @author  MCD Application Team
  * @version V1.0.0
  * @date    22-July-2011
  * @brief   This file provides the high layer firmware functions to manage the 
  *          following functionalities of the USB Audio Class:
  *           - Initialization and Configuration of high and low layer
  *           - Enumeration as Audio Streaming Device
  *           - Audio Streaming data transfer
  *           - AudioControl requests management
  *           - Error management
  *           
  *  @verbatim
  *      
  *          ===================================================================      
  *                                Audio Class Driver Description
  *          =================================================================== 
  *           This driver manages the Audio Class 1.0 following the "USB Device Class Definition for
  *           Audio Devices V1.0 Mar 18, 98".
  *           This driver implements the following aspects of the specification:
  *             - Device descriptor management
  *             - Configuration descriptor management
  *             - Standard AC Interface Descriptor management
  *             - 1 Audio Streaming Interface (with single channel, PCM, Stereo mode)
  *             - 1 Audio Streaming Endpoint
  *             - 1 Audio Terminal Input (1 channel)
  *             - Audio Class-Specific AC Interfaces
  *             - Audio Class-Specific AS Interfaces
  *             - AudioControl Requests: only SET_CUR and GET_CUR requests are supported (for Mute)
  *             - Audio Feature Unit (limited to Mute control)
  *             - Audio Synchronization type: Asynchronous
  *             - Single fixed audio sampling rate (configurable in usbd_conf.h file)
  *          
  *           @note
  *            The Audio Class 1.0 is based on USB Specification 1.0 and thus supports only
  *            Low and Full speed modes and does not allow High Speed transfers.
  *            Please refer to "USB Device Class Definition for Audio Devices V1.0 Mar 18, 98"
  *            for more details.
  * 
  *           These aspects may be enriched or modified for a specific user application.
  *          
  *            This driver doesn't implement the following aspects of the specification 
  *            (but it is possible to manage these features with some modifications on this driver):
  *             - AudioControl Endpoint management
  *             - AudioControl requsests other than SET_CUR and GET_CUR
  *             - Abstraction layer for AudioControl requests (only Mute functionality is managed)
  *             - Audio Synchronization type: Adaptive
  *             - Audio Compression modules and interfaces
  *             - MIDI interfaces and modules
  *             - Mixer/Selector/Processing/Extension Units (Feature unit is limited to Mute control)
  *             - Any other application-specific modules
  *             - Multiple and Variable audio sampling rates
  *             - Out Streaming Endpoint/Interface (microphone)
  *      
  *  @endverbatim
  *                                  
  ******************************************************************************               
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  *
  * <h2><center>&copy; COPYRIGHT 2011 STMicroelectronics</center></h2>
  ******************************************************************************
  */ 

/* Includes ------------------------------------------------------------------*/

#include "usbd_audio_core.h"
#include "usbd_audio_out_if.h"
#include "usbd_usr.h"
#include "usb_regs.h"
#include "usbd_ioreq.h"
#include "stm32f4xx_dma.h"
#include "stm32f4xx_spi.h"
#include "stm32f4xx_tim.h"
#include "stm32f4xx_gpio.h"
#include "stm32f4xx.h"

static volatile uint8_t flag=0;
static volatile uint8_t dpid;

/** @addtogroup STM32_USB_OTG_DEVICE_LIBRARY
  * @{
  */


/** @defgroup usbd_audio 
  * @brief usbd core module
  * @{
  */ 

/** @defgroup usbd_audio_Private_TypesDefinitions
  * @{
  */ 
/**
  * @}
  */ 


/** @defgroup usbd_audio_Private_Defines
  * @{
  */ 
/**
  * @}
  */ 


/** @defgroup usbd_audio_Private_Macros
  * @{
  */ 
/**
  * @}
  */ 


/** @defgroup usbd_audio_Private_FunctionPrototypes
  * @{
  */

/*********************************************
   AUDIO Device library callbacks
 *********************************************/
static uint8_t  usbd_audio_Init       (void  *pdev, uint8_t cfgidx);
static uint8_t  usbd_audio_DeInit     (void  *pdev, uint8_t cfgidx);
static uint8_t  usbd_audio_Setup      (void  *pdev, USB_SETUP_REQ *req);
static uint8_t  usbd_audio_EP0_RxReady(void *pdev);
static uint8_t  usbd_audio_DataIn     (void *pdev, uint8_t epnum);
static uint8_t  usbd_audio_DataOut    (void *pdev, uint8_t epnum);
static uint8_t  usbd_audio_SOF        (void *pdev);
static uint8_t  usbd_audio_IN_Incplt  (void  *pdev);
static uint8_t  usbd_audio_OUT_Incplt (void  *pdev);

/*********************************************
   AUDIO Requests management functions
 *********************************************/
static void AUDIO_Req_GetCurrent      (void *pdev, USB_SETUP_REQ *req);
static void AUDIO_Req_SetCurrent      (void *pdev, USB_SETUP_REQ *req);
static void AUDIO_Req_SetCurrent2	  (USB_SETUP_REQ *req);
static uint8_t *USBD_audio_GetCfgDesc (uint8_t speed, uint16_t *length);
static uint8_t *USBD_audio_GetOtherSpeedCfgDesc ( uint8_t speed, uint16_t *length);
/**
  * @}
  */ 

/** @defgroup usbd_audio_Private_Variables
  * @{
  */ 
/* Main Buffer for Audio Data Out transfers and its relative pointers */
//uint8_t  IsocOutBuff [TOTAL_OUT_BUF_SIZE * 2];
//uint8_t* IsocOutWrPtr = IsocOutBuff;
//uint8_t* IsocOutRdPtr = IsocOutBuff;

#define AUDIO_BUFFER_SIZE           (48*4*32)//32    48
#define SOF_RATE                    0x05
#define USB_BUFFER_SIZE             ((192+1)*2*2)
#define FB_RATE_DELTA               (1<<12)
#define FB_RATE_DELTA_NUM           4

#define SAMPLING_FREQ_MUTE_CONTROL  0x01
#define ENDPOINT_REQUEST_TYPE       0x02
#define RESERVED_MASK               ((uint32_t)0x0F7D0F7D)

/* Main Buffer for Audio Control Rrequests transfers and its relative variables */
volatile uint8_t  AudioCtl[64] __attribute__((aligned(4)));
uint8_t  AudioCtlCmd = 0;
uint32_t AudioCtlLen = 0;
uint8_t  AudioCtlUnit = 0;
uint16_t Audio_buf_0[AUDIO_BUFFER_SIZE];
uint16_t Audio_buf_1[AUDIO_BUFFER_SIZE];
volatile uint32_t Sampling_freq __attribute__((aligned(4))) =44100;
uint8_t Speed=0;
uint8_t mute=0;
uint32_t mult=1;
uint16_t gap;
uint16_t old_gap;

uint32_t FB_NOM=(44<<14)+(1<<14)/10;//  44,1//(44<<14)+(1<<14)/10;//  44,1
uint32_t FB_RATE;
int delta_num=0;
uint16_t num_remaining;
uint16_t num_remaining_old=0;
uint8_t tmpbuf[USB_BUFFER_SIZE];
uint8_t spk_buffer_in=0;
uint8_t spk_buffer_out=0;
uint16_t spk_index=0;
uint8_t out_stop=0;
static uint32_t PlayStart = 0;
static volatile  uint16_t SOF_num=0;
static __IO uint32_t  usbd_audio_AltSet = 0;
static uint8_t usbd_audio_CfgDesc[AUDIO_CONFIG_DESC_SIZE];

typedef struct usb_device_qualifier_descriptor {
           uint8_t bLength;
           uint8_t bDescriptorType;
           uint16_t bcdUSB;
           uint8_t bDeviceClass;
           uint8_t bDeviceSubClass;
           uint8_t bDeviceProtocol;
           uint8_t bMaxPacketSize0;
           uint8_t bNumConfigurations;
           uint8_t bReserved;
   } __attribute__((packed));

   struct usb_device_qualifier_descriptor usb_device_qualifier_desc=
   {
		   .bLength=10,
           .bDescriptorType=USB_DT_DEVICE_QUALIFIER,
           .bcdUSB=0x0200,
           .bDeviceClass=0,
           .bDeviceSubClass=0,
           .bDeviceProtocol=0,
           .bMaxPacketSize0=64,
           .bNumConfigurations=1,
           .bReserved=0
   };

   void SetI2SMCLK(uint32_t freq) {
   	const uint32_t I2SFREQ[] = { 44100, 48000, 88200, 96000 , 176400, 192000  };
   	const uint32_t PLLI2SN[] = { 271, 295, 271, 295, 361, 393 };
   	const uint32_t PLLI2SR[] = { 6, 4, 3, 2, 2,2  };

   	const uint32_t I2SDIV[] = {2,3,2,3,2,2 };
   	const uint32_t I2SODD[] = {0,0,0,0,0,0 };

   	uint8_t index = 0;
   	while ((freq != I2SFREQ[index]) && index < 7)
   		index++;
   	if (index==6) return;

   	/******************************************************************************/
   	/*                        I2S clock configuration                             */
   	/******************************************************************************/
   //External clock is used as i2s master clk
   	RCC->CR &= ~((uint32_t)RCC_CR_PLLI2SON);

   	/* Configure PLLI2S */
   	RCC->PLLI2SCFGR = (PLLI2SN[index] << 6) | (PLLI2SR[index] << 28);
   //	/* Enable PLLI2S */
   	RCC->CR |= ((uint32_t)RCC_CR_PLLI2SON);
   	/* Wait till PLLI2S is ready */
   	while((RCC->CR & RCC_CR_PLLI2SRDY) == 0)
   	 {
   	 }
   	/* PLLI2S clock used as I2S clock source */
   	RCC_I2SCLKConfig(RCC_I2S2CLKSource_PLLI2S);// AUDIO CLOCK
   	//RCC->CFGR |= RCC_CFGR_I2SSRC;

   	/* Write to SPIx I2SPR register the computed value */
   	SPI3->I2SPR = (uint16_t)((uint16_t) I2SDIV[index] | (uint16_t)((I2SODD[index] << 8)) | (uint16_t)(I2S_MCLKOutput_Enable));

   }


/* AUDIO interface class callbacks structure */
USBD_Class_cb_TypeDef  AUDIO_cb = 
{
  usbd_audio_Init,
  usbd_audio_DeInit,
  usbd_audio_Setup,
  NULL, /* EP0_TxSent */
  usbd_audio_EP0_RxReady,
  usbd_audio_DataIn,
  usbd_audio_DataOut,
  usbd_audio_SOF,
  usbd_audio_IN_Incplt,
  usbd_audio_OUT_Incplt,   
  USBD_audio_GetCfgDesc,
  USBD_audio_GetOtherSpeedCfgDesc
};

/* USB AUDIO device Configuration Descriptor */
static uint8_t usbd_audio_CfgDesc[AUDIO_CONFIG_DESC_SIZE] =
{
  /* Configuration 1 */
  USB_CONFIGUARTION_DESC_SIZE,          /* bLength */
  USB_CONFIGURATION_DESCRIPTOR_TYPE,    /* bDescriptorType */
  LOBYTE(AUDIO_CONFIG_DESC_SIZE),       /* wTotalLength  109 bytes*/
  HIBYTE(AUDIO_CONFIG_DESC_SIZE),      
  0x02,                                 /* bNumInterfaces */
  0x01,                                 /* bConfigurationValue */
  0x00,                                 /* iConfiguration */
  0xC0,                                 /* bmAttributes  BUS Powred*/
  0x32,                                 /* bMaxPower = 100 mA*/
  /* 09 byte*/
  
  /* USB Speaker Standard interface descriptor */
  AUDIO_INTERFACE_DESC_SIZE,            /* bLength */
  USB_INTERFACE_DESCRIPTOR_TYPE,        /* bDescriptorType */
  0x00,                                 /* bInterfaceNumber */
  0x00,                                 /* bAlternateSetting */
  0x00,                                 /* bNumEndpoints */
  USB_DEVICE_CLASS_AUDIO,               /* bInterfaceClass */
  AUDIO_SUBCLASS_AUDIOCONTROL,          /* bInterfaceSubClass */
  AUDIO_PROTOCOL_UNDEFINED,             /* bInterfaceProtocol */
  0x04,                                 /* iInterface */
  /* 09 byte*/
  
  /* USB Speaker Class-specific AC Interface Descriptor */
  AUDIO_INTERFACE_DESC_SIZE,            /* bLength */
  AUDIO_INTERFACE_DESCRIPTOR_TYPE,      /* bDescriptorType */
  AUDIO_CONTROL_HEADER,                 /* bDescriptorSubtype */
  0x00,          /* 1.00 */             /* bcdADC */
  0x01,
  0x28,                                 /* wTotalLength = 40*/
  0x00,
  0x01,                                 /* bInCollection */
  0x01,                                 /* baInterfaceNr */
  /* 09 byte*/
  
  /* USB Speaker Input Terminal Descriptor */
  AUDIO_INPUT_TERMINAL_DESC_SIZE,       /* bLength */
  AUDIO_INTERFACE_DESCRIPTOR_TYPE,      /* bDescriptorType */
  AUDIO_CONTROL_INPUT_TERMINAL,         /* bDescriptorSubtype */
  0x01,                                 /* bTerminalID */
  0x01,                                 /* wTerminalType AUDIO_TERMINAL_USB_STREAMING   0x0101 */
  0x01,
  0x00,                                 /* bAssocTerminal */
  0x02,                                 /* bNrChannels */
  0x03,                                 /* wChannelConfig 0x0003  FL FR */
  0x00,
  0x00,                                 /* iChannelNames */
  0x00,                                 /* iTerminal */
  /* 12 byte*/
  
  /* USB Speaker Audio Feature Unit Descriptor */
  AUDIO_FEATURE_UNIT_DESC_SZ+1,           /* bLength */
  AUDIO_INTERFACE_DESCRIPTOR_TYPE,      /* bDescriptorType */
  AUDIO_CONTROL_FEATURE_UNIT,           /* bDescriptorSubtype */
  AUDIO_OUT_STREAMING_CTRL,             /* bUnitID */
  0x01,                                 /* bSourceID */
  0x01,                                 /* bControlSize */
  AUDIO_CONTROL_MUTE,                   /* bmaControls(0) */
  0x00,                                 /* bmaControls(1) */
  0x00,                                 /* bmaControls(2) */
  0x00,                                 /* iTerminal */
  /* 09 byte*/
  
  /*USB Speaker Output Terminal Descriptor */
  0x09,      /* bLength */
  AUDIO_INTERFACE_DESCRIPTOR_TYPE,      /* bDescriptorType */
  AUDIO_CONTROL_OUTPUT_TERMINAL,        /* bDescriptorSubtype */
  0x03,                                 /* bTerminalID */
  0x01,                                 /* wTerminalType  0x0301*/
  0x03,
  0x00,                                 /* bAssocTerminal */
  0x02,                                 /* bSourceID */
  0x00,                                 /* iTerminal */
  /* 09 byte*/
  
  /* USB Speaker Standard AS Interface Descriptor - Audio Streaming Zero Bandwith */
  /* Interface 1, Alternate Setting 0                                             */
  AUDIO_INTERFACE_DESC_SIZE,            /* bLength */
  USB_INTERFACE_DESCRIPTOR_TYPE,        /* bDescriptorType */
  0x01,                                 /* bInterfaceNumber */
  0x00,                                 /* bAlternateSetting */
  0x00,                                 /* bNumEndpoints */
  USB_DEVICE_CLASS_AUDIO,               /* bInterfaceClass */
  AUDIO_SUBCLASS_AUDIOSTREAMING,        /* bInterfaceSubClass */
  AUDIO_PROTOCOL_UNDEFINED,             /* bInterfaceProtocol */
  0x05,                                 /* iInterface */
  /* 09 byte*/
  
  /* USB Speaker Standard AS Interface Descriptor - Audio Streaming Operational */
  /* Interface 1, Alternate Setting 1                                           */
  AUDIO_INTERFACE_DESC_SIZE,            /* bLength */
  USB_INTERFACE_DESCRIPTOR_TYPE,        /* bDescriptorType */
  0x01,                                 /* bInterfaceNumber */
  0x01,                                 /* bAlternateSetting */
  0x02,                                 /* bNumEndpoints */
  USB_DEVICE_CLASS_AUDIO,               /* bInterfaceClass */
  AUDIO_SUBCLASS_AUDIOSTREAMING,        /* bInterfaceSubClass */
  AUDIO_PROTOCOL_UNDEFINED,             /* bInterfaceProtocol */
  0x05,                                 /* iInterface */
  /* 09 byte*/
  
  /* USB Speaker Audio Streaming Interface Descriptor */
  AUDIO_STREAMING_INTERFACE_DESC_SIZE,  /* bLength */
  AUDIO_INTERFACE_DESCRIPTOR_TYPE,      /* bDescriptorType */
  AUDIO_STREAMING_GENERAL,              /* bDescriptorSubtype */
  0x01,                                 /* bTerminalLink */
  0x04,                                 /* bDelay */
  0x01,                                 /* wFormatTag AUDIO_FORMAT_PCM  0x0001*/
  0x00,
  /* 07 byte*/
  
  /* USB Speaker Audio Type III Format Interface Descriptor */
  AUDIO_FORMAT_TYPE_III_DESC_SZ+5*3,        /* bLength */
  AUDIO_INTERFACE_DESCRIPTOR_TYPE,      /* bDescriptorType */
  AUDIO_STREAMING_FORMAT_TYPE,          /* bDescriptorSubtype */
  AUDIO_FORMAT_TYPE_I,                  /* bFormatType */
  0x02,                                 /* bNrChannels */
  0x02,                                 /* bSubFrameSize :  2 Bytes per frame (16bits) */
  0x10,                                 /* bBitResolution (16-bits per sample) */
  0x06,                                 /* bSamFreqType: 4 frequencies supported */
  SAMPLE_FREQ(44100),                   /* Audio sampling frequency coded on 3 bytes */
  SAMPLE_FREQ(48000),
  SAMPLE_FREQ(88200),
  SAMPLE_FREQ(96000),
  SAMPLE_FREQ(176400),
  SAMPLE_FREQ(192000),
  /* 20 byte*/
  
  /* Endpoint 1 - Standard Descriptor */
  AUDIO_STANDARD_ENDPOINT_DESC_SIZE,    /* bLength */
  USB_ENDPOINT_DESCRIPTOR_TYPE,         /* bDescriptorType */
  AUDIO_OUT_EP,                         /* bEndpointAddress 1 out endpoint*/
  0x05,                                 /* bmAttributes */
  0x04,0x3,                 			/* wMaxPacketSize in Bytes (Freq(Samples+2)*2(Stereo)*2(16bit)) 200*/
  0x04,                                 /* bInterval */
  0x00,                                 /* bRefresh */
  AUDIO_IN_EP,                          /* bSynchAddress */
  /* 09 byte*/
  
  /* Endpoint - Audio Streaming Descriptor*/
  AUDIO_STREAMING_ENDPOINT_DESC_SIZE,   /* bLength */
  AUDIO_ENDPOINT_DESCRIPTOR_TYPE,       /* bDescriptorType */
  AUDIO_ENDPOINT_GENERAL,               /* bDescriptor */
  0x01,                                 /* bmAttributes - Sample Freq Control*/
  0x00,                                 /* bLockDelayUnits */
  0x00,                                 /* wLockDelay */
  0x00,
  /* 07 byte*/

  /* Endpoint 2 - Standard Descriptor */
  AUDIO_STANDARD_ENDPOINT_DESC_SIZE,    /* bLength */
  USB_ENDPOINT_DESCRIPTOR_TYPE,         /* bDescriptorType */
  AUDIO_IN_EP,                          /* bEndpointAddress 2 in endpoint*/
  0x11,        						    /* bmAttributes */
  3,0,   						        /* wMaxPacketSize in Bytes 3 */
  4,								    /* bInterval 1ms*/
  SOF_RATE,							    /* bRefresh 32ms*/
  0x00,                                 /* bSynchAddress */
  /* 09 byte*/

  /* Interface 1, Alternate Setting 2                                          */
  AUDIO_INTERFACE_DESC_SIZE,            /* bLength */
  USB_INTERFACE_DESCRIPTOR_TYPE,        /* bDescriptorType */
  0x01,                                 /* bInterfaceNumber */
  0x02,                                 /* bAlternateSetting */
  0x02,                                 /* bNumEndpoints */
  USB_DEVICE_CLASS_AUDIO,               /* bInterfaceClass */
  AUDIO_SUBCLASS_AUDIOSTREAMING,        /* bInterfaceSubClass */
  AUDIO_PROTOCOL_UNDEFINED,             /* bInterfaceProtocol */
  0x05,                                 /* iInterface */
  /* 09 byte*/

  /* USB Speaker Audio Streaming Interface Descriptor */
  AUDIO_STREAMING_INTERFACE_DESC_SIZE,  /* bLength */
  AUDIO_INTERFACE_DESCRIPTOR_TYPE,      /* bDescriptorType */
  AUDIO_STREAMING_GENERAL,              /* bDescriptorSubtype */
  0x01,                                 /* bTerminalLink */
  0x04,                                 /* bDelay */
  0x01,                                 /* wFormatTag AUDIO_FORMAT_PCM  0x0001*/
  0x00,
  /* 07 byte*/

  /* USB Speaker Audio Type III Format Interface Descriptor */
  AUDIO_FORMAT_TYPE_III_DESC_SZ+3*3,        /* bLength */
  AUDIO_INTERFACE_DESCRIPTOR_TYPE,      /* bDescriptorType */
  AUDIO_STREAMING_FORMAT_TYPE,          /* bDescriptorSubtype */
  AUDIO_FORMAT_TYPE_I,                  /* bFormatType */
  0x02,                                 /* bNrChannels */
  0x03,                                 /* bSubFrameSize :  3 Bytes per frame (48bits) */
  0x18,                                 /* bBitResolution (24-bits per sample) */
  0x04,                                 /* bSamFreqType: 4 frequencies supported */
  SAMPLE_FREQ(44100),                   /* Audio sampling frequency coded on 3 bytes */
  SAMPLE_FREQ(48000),
  SAMPLE_FREQ(88200),
  SAMPLE_FREQ(96000),
  /* 20 byte*/

  /* Endpoint 1 - Standard Descriptor */
  AUDIO_STANDARD_ENDPOINT_DESC_SIZE,    /* bLength */
  USB_ENDPOINT_DESCRIPTOR_TYPE,         /* bDescriptorType */
  AUDIO_OUT_EP,                         /* bEndpointAddress 1 out endpoint*/
  0x05,                                 /* bmAttributes */
  0x46,0x02,                 			/* wMaxPacketSize in Bytes (Freq(Samples+2)*2(Stereo)*2(16bit)) 200*/
  0x04,                                 /* bInterval */
  0x00,                                 /* bRefresh */
  AUDIO_IN_EP,                          /* bSynchAddress */
  /* 09 byte*/

  /* Endpoint - Audio Streaming Descriptor*/
  AUDIO_STREAMING_ENDPOINT_DESC_SIZE,   /* bLength */
  AUDIO_ENDPOINT_DESCRIPTOR_TYPE,       /* bDescriptorType */
  AUDIO_ENDPOINT_GENERAL,               /* bDescriptor */
  0x01,                                 /* bmAttributes - Sample Freq Control*/
  0x00,                                 /* bLockDelayUnits */
  0x00,                                 /* wLockDelay */
  0x00,
  /* 07 byte*/

  /* Endpoint 2 - Standard Descriptor */
  AUDIO_STANDARD_ENDPOINT_DESC_SIZE,    /* bLength */
  USB_ENDPOINT_DESCRIPTOR_TYPE,         /* bDescriptorType */
  AUDIO_IN_EP,                          /* bEndpointAddress 2 in endpoint*/
  0x11,        						    /* bmAttributes */
  3,0,   						        /* wMaxPacketSize in Bytes 3 */
  4,								    /* bInterval 1ms*/
  SOF_RATE,							    /* bRefresh 32ms*/
  0x00,                                 /* bSynchAddress */
  /* 09 byte*/


} ;

/* USB AUDIO device Configuration Descriptor */
static uint8_t usbd_audio_FSCfgDesc[AUDIO_CONFIG_DESC_SIZE] =
{
  /* Configuration 1 */
  USB_CONFIGUARTION_DESC_SIZE,          /* bLength */
  USB_CONFIGURATION_DESCRIPTOR_TYPE,    /* bDescriptorType */
  LOBYTE(AUDIO_CONFIG_DESC_SIZE),       /* wTotalLength  109 bytes*/
  HIBYTE(AUDIO_CONFIG_DESC_SIZE),
  0x02,                                 /* bNumInterfaces */
  0x01,                                 /* bConfigurationValue */
  0x00,                                 /* iConfiguration */
  0xC0,                                 /* bmAttributes  BUS Powred*/
  0x32,                                 /* bMaxPower = 100 mA*/
  /* 09 byte*/

  /* USB Speaker Standard interface descriptor */
  AUDIO_INTERFACE_DESC_SIZE,            /* bLength */
  USB_INTERFACE_DESCRIPTOR_TYPE,        /* bDescriptorType */
  0x00,                                 /* bInterfaceNumber */
  0x00,                                 /* bAlternateSetting */
  0x00,                                 /* bNumEndpoints */
  USB_DEVICE_CLASS_AUDIO,               /* bInterfaceClass */
  AUDIO_SUBCLASS_AUDIOCONTROL,          /* bInterfaceSubClass */
  AUDIO_PROTOCOL_UNDEFINED,             /* bInterfaceProtocol */
  0x04,                                 /* iInterface */
  /* 09 byte*/

  /* USB Speaker Class-specific AC Interface Descriptor */
  AUDIO_INTERFACE_DESC_SIZE,            /* bLength */
  AUDIO_INTERFACE_DESCRIPTOR_TYPE,      /* bDescriptorType */
  AUDIO_CONTROL_HEADER,                 /* bDescriptorSubtype */
  0x00,          /* 1.00 */             /* bcdADC */
  0x01,
  0x28,                                 /* wTotalLength = 40*/
  0x00,
  0x01,                                 /* bInCollection */
  0x01,                                 /* baInterfaceNr */
  /* 09 byte*/

  /* USB Speaker Input Terminal Descriptor */
  AUDIO_INPUT_TERMINAL_DESC_SIZE,       /* bLength */
  AUDIO_INTERFACE_DESCRIPTOR_TYPE,      /* bDescriptorType */
  AUDIO_CONTROL_INPUT_TERMINAL,         /* bDescriptorSubtype */
  0x01,                                 /* bTerminalID */
  0x01,                                 /* wTerminalType AUDIO_TERMINAL_USB_STREAMING   0x0101 */
  0x01,
  0x00,                                 /* bAssocTerminal */
  0x02,                                 /* bNrChannels */
  0x03,                                 /* wChannelConfig 0x0003  FL FR */
  0x00,
  0x00,                                 /* iChannelNames */
  0x00,                                 /* iTerminal */
  /* 12 byte*/

  /* USB Speaker Audio Feature Unit Descriptor */
  AUDIO_FEATURE_UNIT_DESC_SZ+1,           /* bLength */
  AUDIO_INTERFACE_DESCRIPTOR_TYPE,      /* bDescriptorType */
  AUDIO_CONTROL_FEATURE_UNIT,           /* bDescriptorSubtype */
  AUDIO_OUT_STREAMING_CTRL,             /* bUnitID */
  0x01,                                 /* bSourceID */
  0x01,                                 /* bControlSize */
  AUDIO_CONTROL_MUTE,                   /* bmaControls(0) */
  0x00,                                 /* bmaControls(1) */
  0x00,                                 /* bmaControls(2) */
  0x00,                                 /* iTerminal */
  /* 09 byte*/

  /*USB Speaker Output Terminal Descriptor */
  0x09,      /* bLength */
  AUDIO_INTERFACE_DESCRIPTOR_TYPE,      /* bDescriptorType */
  AUDIO_CONTROL_OUTPUT_TERMINAL,        /* bDescriptorSubtype */
  0x03,                                 /* bTerminalID */
  0x01,                                 /* wTerminalType  0x0301*/
  0x03,
  0x00,                                 /* bAssocTerminal */
  0x02,                                 /* bSourceID */
  0x00,                                 /* iTerminal */
  /* 09 byte*/

  /* USB Speaker Standard AS Interface Descriptor - Audio Streaming Zero Bandwith */
  /* Interface 1, Alternate Setting 0                                             */
  AUDIO_INTERFACE_DESC_SIZE,            /* bLength */
  USB_INTERFACE_DESCRIPTOR_TYPE,        /* bDescriptorType */
  0x01,                                 /* bInterfaceNumber */
  0x00,                                 /* bAlternateSetting */
  0x00,                                 /* bNumEndpoints */
  USB_DEVICE_CLASS_AUDIO,               /* bInterfaceClass */
  AUDIO_SUBCLASS_AUDIOSTREAMING,        /* bInterfaceSubClass */
  AUDIO_PROTOCOL_UNDEFINED,             /* bInterfaceProtocol */
  0x05,                                 /* iInterface */
  /* 09 byte*/

  /* USB Speaker Standard AS Interface Descriptor - Audio Streaming Operational */
  /* Interface 1, Alternate Setting 1                                           */
  AUDIO_INTERFACE_DESC_SIZE,            /* bLength */
  USB_INTERFACE_DESCRIPTOR_TYPE,        /* bDescriptorType */
  0x01,                                 /* bInterfaceNumber */
  0x01,                                 /* bAlternateSetting */
  0x02,                                 /* bNumEndpoints */
  USB_DEVICE_CLASS_AUDIO,               /* bInterfaceClass */
  AUDIO_SUBCLASS_AUDIOSTREAMING,        /* bInterfaceSubClass */
  AUDIO_PROTOCOL_UNDEFINED,             /* bInterfaceProtocol */
  0x05,                                 /* iInterface */
  /* 09 byte*/

  /* USB Speaker Audio Streaming Interface Descriptor */
  AUDIO_STREAMING_INTERFACE_DESC_SIZE,  /* bLength */
  AUDIO_INTERFACE_DESCRIPTOR_TYPE,      /* bDescriptorType */
  AUDIO_STREAMING_GENERAL,              /* bDescriptorSubtype */
  0x01,                                 /* bTerminalLink */
  0x01,                                 /* bDelay */
  0x01,                                 /* wFormatTag AUDIO_FORMAT_PCM  0x0001*/
  0x00,
  /* 07 byte*/

  /* USB Speaker Audio Type III Format Interface Descriptor */
  AUDIO_FORMAT_TYPE_III_DESC_SZ+5*3,        /* bLength */
  AUDIO_INTERFACE_DESCRIPTOR_TYPE,      /* bDescriptorType */
  AUDIO_STREAMING_FORMAT_TYPE,          /* bDescriptorSubtype */
  AUDIO_FORMAT_TYPE_I,                  /* bFormatType */
  0x02,                                 /* bNrChannels */
  0x02,                                 /* bSubFrameSize :  2 Bytes per frame (16bits) */
  0x10,                                 /* bBitResolution (16-bits per sample) */
  0x06,                                 /* bSamFreqType: 4 frequencies supported */
  SAMPLE_FREQ(44100),                   /* Audio sampling frequency coded on 3 bytes */
  SAMPLE_FREQ(48000),
  SAMPLE_FREQ(88200),
  SAMPLE_FREQ(96000),
  SAMPLE_FREQ(176400),
  SAMPLE_FREQ(192000),
  /* 20 byte*/

  /* Endpoint 1 - Standard Descriptor */
  AUDIO_STANDARD_ENDPOINT_DESC_SIZE,    /* bLength */
  USB_ENDPOINT_DESCRIPTOR_TYPE,         /* bDescriptorType */
  AUDIO_OUT_EP,                         /* bEndpointAddress 1 out endpoint*/
  0x05,                                 /* bmAttributes */
  0x04,0x3,                 			/* wMaxPacketSize in Bytes (Freq(Samples+2)*2(Stereo)*2(16bit)) 200*/
  0x01,                                 /* bInterval */
  0x00,                                 /* bRefresh */
  AUDIO_IN_EP,                          /* bSynchAddress */
  /* 09 byte*/

  /* Endpoint - Audio Streaming Descriptor*/
  AUDIO_STREAMING_ENDPOINT_DESC_SIZE,   /* bLength */
  AUDIO_ENDPOINT_DESCRIPTOR_TYPE,       /* bDescriptorType */
  AUDIO_ENDPOINT_GENERAL,               /* bDescriptor */
  0x01,                                 /* bmAttributes - Sample Freq Control*/
  0x00,                                 /* bLockDelayUnits */
  0x00,                                 /* wLockDelay */
  0x00,
  /* 07 byte*/

  /* Endpoint 2 - Standard Descriptor */
  AUDIO_STANDARD_ENDPOINT_DESC_SIZE,    /* bLength */
  USB_ENDPOINT_DESCRIPTOR_TYPE,         /* bDescriptorType */
  AUDIO_IN_EP,                          /* bEndpointAddress 2 in endpoint*/
  0x11,        						    /* bmAttributes */
  3,0,   						        /* wMaxPacketSize in Bytes 3 */
  1,								    /* bInterval 1ms*/
  SOF_RATE,							    /* bRefresh 32ms*/
  0x00,                                 /* bSynchAddress */
  /* 09 byte*/

  /* Interface 1, Alternate Setting 2                                          */
  AUDIO_INTERFACE_DESC_SIZE,            /* bLength */
  USB_INTERFACE_DESCRIPTOR_TYPE,        /* bDescriptorType */
  0x01,                                 /* bInterfaceNumber */
  0x02,                                 /* bAlternateSetting */
  0x02,                                 /* bNumEndpoints */
  USB_DEVICE_CLASS_AUDIO,               /* bInterfaceClass */
  AUDIO_SUBCLASS_AUDIOSTREAMING,        /* bInterfaceSubClass */
  AUDIO_PROTOCOL_UNDEFINED,             /* bInterfaceProtocol */
  0x05,                                 /* iInterface */
  /* 09 byte*/

  /* USB Speaker Audio Streaming Interface Descriptor */
  AUDIO_STREAMING_INTERFACE_DESC_SIZE,  /* bLength */
  AUDIO_INTERFACE_DESCRIPTOR_TYPE,      /* bDescriptorType */
  AUDIO_STREAMING_GENERAL,              /* bDescriptorSubtype */
  0x01,                                 /* bTerminalLink */
  0x04,                                 /* bDelay */
  0x01,                                 /* wFormatTag AUDIO_FORMAT_PCM  0x0001*/
  0x00,
  /* 07 byte*/

  /* USB Speaker Audio Type III Format Interface Descriptor */
  AUDIO_FORMAT_TYPE_III_DESC_SZ+3*3,        /* bLength */
  AUDIO_INTERFACE_DESCRIPTOR_TYPE,      /* bDescriptorType */
  AUDIO_STREAMING_FORMAT_TYPE,          /* bDescriptorSubtype */
  AUDIO_FORMAT_TYPE_I,                  /* bFormatType */
  0x02,                                 /* bNrChannels */
  0x03,                                 /* bSubFrameSize :  3 Bytes per frame (48bits) */
  0x18,                                 /* bBitResolution (24-bits per sample) */
  0x04,                                 /* bSamFreqType: 4 frequencies supported */
  SAMPLE_FREQ(44100),                   /* Audio sampling frequency coded on 3 bytes */
  SAMPLE_FREQ(48000),
  SAMPLE_FREQ(88200),
  SAMPLE_FREQ(96000),
  /* 20 byte*/

  /* Endpoint 1 - Standard Descriptor */
  AUDIO_STANDARD_ENDPOINT_DESC_SIZE,    /* bLength */
  USB_ENDPOINT_DESCRIPTOR_TYPE,         /* bDescriptorType */
  AUDIO_OUT_EP,                         /* bEndpointAddress 1 out endpoint*/
  0x05,                                 /* bmAttributes */
  0x46,0x02,                 			/* wMaxPacketSize in Bytes (Freq(Samples+2)*2(Stereo)*2(16bit)) 200*/
  0x01,                                 /* bInterval */
  0x00,                                 /* bRefresh */
  AUDIO_IN_EP,                          /* bSynchAddress */
  /* 09 byte*/

  /* Endpoint - Audio Streaming Descriptor*/
  AUDIO_STREAMING_ENDPOINT_DESC_SIZE,   /* bLength */
  AUDIO_ENDPOINT_DESCRIPTOR_TYPE,       /* bDescriptorType */
  AUDIO_ENDPOINT_GENERAL,               /* bDescriptor */
  0x01,                                 /* bmAttributes - Sample Freq Control*/
  0x00,                                 /* bLockDelayUnits */
  0x00,                                 /* wLockDelay */
  0x00,
  /* 07 byte*/

  /* Endpoint 2 - Standard Descriptor */
  AUDIO_STANDARD_ENDPOINT_DESC_SIZE,    /* bLength */
  USB_ENDPOINT_DESCRIPTOR_TYPE,         /* bDescriptorType */
  AUDIO_IN_EP,                          /* bEndpointAddress 2 in endpoint*/
  0x11,        						    /* bmAttributes */
  3,0,   						        /* wMaxPacketSize in Bytes 3 */
  1,								    /* bInterval 1ms*/
  SOF_RATE,							    /* bRefresh 32ms*/
  0x00,                                 /* bSynchAddress */
  /* 09 byte*/
} ;

//uint32_t feedback_data;
//uint32_t FB_nom=(44<<14);
//uint8_t OUT_num=0;
//uint8_t IN_num=0;

/**
  * @}
  */ 

/** @defgroup usbd_audio_Private_Functions
  * @{
  */ 

/**
* @brief  usbd_audio_Init
*         Initilaizes the AUDIO interface.
* @param  pdev: device instance
* @param  cfgidx: Configuration index
* @retval status
*/
static uint8_t  usbd_audio_Init (void  *pdev, 
                                 uint8_t cfgidx)
{  
	  /* Open EP OUT */
	  DCD_EP_Open(pdev,
	              AUDIO_OUT_EP,
	              USB_BUFFER_SIZE,
	              USB_OTG_EP_ISOC);

	  /* Open EP IN */
	  DCD_EP_Open(pdev,
			  	  AUDIO_IN_EP,
	              3,
	              USB_OTG_EP_ISOC);

	  DCD_EP_Flush(pdev,AUDIO_IN_EP);


	  /* Initialize the Audio output Hardware layer */
	  if (AUDIO_OUT_fops.Init(USBD_AUDIO_FREQ) != USBD_OK)
	  {
		  return USBD_FAIL;
	  }
	  flag=1;
//	  GPIO_SetBits(GPIOD,GPIO_Pin_15);
	  return USBD_OK;
}

/**
* @brief  usbd_audio_DeInit
*         DeInitializes the AUDIO layer.
* @param  pdev: device instance
* @param  cfgidx: Configuration index
* @retval status
*/
static uint8_t  usbd_audio_DeInit (void  *pdev, 
                                   uint8_t cfgidx)
{ 
	  DCD_EP_Flush (pdev , AUDIO_OUT_EP);
	  DCD_EP_Flush (pdev , AUDIO_IN_EP);
	  DCD_EP_Close (pdev , AUDIO_OUT_EP);
	  DCD_EP_Close (pdev , AUDIO_IN_EP);
	  flag = 0;

	  /* DeInitialize the Audio output Hardware layer */
	  if (AUDIO_OUT_fops.DeInit(0) != USBD_OK)
	  {
	    return USBD_FAIL;
	  }
//	  GPIO_ResetBits(GPIOD,GPIO_Pin_15);
	  return USBD_OK;
}

/**
  * @brief  usbd_audio_Setup
  *         Handles the Audio control request parsing.
  * @param  pdev: instance
  * @param  req: usb requests
  * @retval status
  */
static uint8_t  usbd_audio_Setup (void  *pdev, 
                                  USB_SETUP_REQ *req)
{
	  uint16_t len;
	  uint8_t  *pbuf;
	//  uint32_t tmp;

	  switch (req->bmRequest & USB_REQ_TYPE_MASK)
	  {
	    /* AUDIO Class Requests -------------------------------*/
	  	  case USB_REQ_TYPE_CLASS :
	  		  switch (req->bRequest)
	  		  {
	  		  	  case AUDIO_REQ_GET_CUR:
	  		  		  AUDIO_Req_GetCurrent(pdev, req);
	  		  		  break;

	  		  	  case AUDIO_REQ_SET_CUR:
	  		  		  AUDIO_Req_SetCurrent(pdev, req);
	  		  		  break;

	  		  	  default:
	  		  		  USBD_CtlError (pdev, req);
	  		  		  return USBD_FAIL;
	  		  }
	  		  break;

	    /* Standard Requests -------------------------------*/
	  	  case USB_REQ_TYPE_STANDARD:
	  		  switch (req->bRequest)
	  		  {
	  		  	  case USB_REQ_GET_DESCRIPTOR:
	  		  		  if ( (req->wValue >> 8) == AUDIO_DESCRIPTOR_TYPE)
	  		  		  {
	  		  			  pbuf = usbd_audio_CfgDesc+18; //+18
	  		  			  len = MIN(USB_AUDIO_DESC_SIZ , req->wLength);
	  		  		  }

	  		  		 if ( (req->wValue >> 8) == USB_DT_DEVICE_QUALIFIER)
	  		  			  		  		  {
	  		  			  		  			  pbuf = USBD_DeviceQualifierDesc;
	  		  			  		  			  len = MIN(USB_LEN_DEV_QUALIFIER_DESC , req->wLength);
	  		  			  		  		  }

	  		  		  USBD_CtlSendData (pdev,
	  		  				  	  	  	pbuf,
	                                    len);
	  		  		  break;

	  		  	  case USB_REQ_GET_INTERFACE :
	  		  		  USBD_CtlSendData (pdev,
	                                   (uint8_t *)&usbd_audio_AltSet,
	                                    1);
	  		  		  break;

	  		  	  case USB_REQ_SET_INTERFACE :
	  		  		  if ((uint8_t)(req->wValue) < AUDIO_TOTAL_IF_NUM)
	  		  		  {
	  		  			  usbd_audio_AltSet = (uint8_t)(req->wValue);
	  		  			  if (usbd_audio_AltSet == 2)
	  		  			  {
	  		  				  SOF_num = 0;
	  		  				  flag = 0;
	  		  				  DCD_EP_Flush(pdev,AUDIO_IN_EP);
	  		  				GPIOF->ODR|=GPIO_Pin_6;
	  		  			  }
	  		  			  if (usbd_audio_AltSet == 1)
	  		  			  {
	  		  				  SOF_num = 0;
	  		  				  flag = 0;
	  		  				  DCD_EP_Flush(pdev,AUDIO_IN_EP);
		  		  				GPIOF->ODR|=GPIO_Pin_6;
	  		  			  }
	  		  			  if (usbd_audio_AltSet == 0)
	  		  			  {
	  		  				  flag = 0;
	  		  				  DCD_EP_Flush(pdev,AUDIO_IN_EP);
		  		  				GPIOF->ODR&=~GPIO_Pin_6;
	  		  			  }
	  		  		  }
	  		  		  else
	  		  		  {
	  		  			  /* Call the error management function (command will be nacked */
	  		  			  USBD_CtlError (pdev, req);
	  		  		  }
	  		  		  break;
	  		  }
	  }
	  return USBD_OK;
}


/**
  * @brief  usbd_audio_DataIn
  *         Handles the audio IN data stage.
  * @param  pdev: instance
  * @param  epnum: endpoint number
  * @retval status
  */
static uint8_t  usbd_audio_DataIn (void *pdev, uint8_t epnum)
{
	if (epnum == (AUDIO_IN_EP&0x7f))
	{
		//GPIO_ToggleBits(GPIOC,GPIO_Pin_1);
		flag = 0;
		//SOF_num=0;
		//DCD_EP_Tx (pdev, AUDIO_IN_EP, (uint8_t *) &FB_RATE, 3);
	}
	return USBD_OK;
}

/**
  * @brief  usbd_audio_DataOut
  *         Handles the Audio Out data stage.
  * @param  pdev: instance
  * @param  epnum: endpoint number
  * @retval status
  */
static uint8_t  usbd_audio_DataOut (void *pdev, uint8_t epnum)
{     
	  uint16_t num_samples,num_byte,k,l;
	  uint8_t byte_per_sampl;

	if (epnum == AUDIO_OUT_EP)
	{
		GPIOF->ODR^=GPIO_Pin_8;
		num_byte = USBD_GetRxCount (pdev,epnum);
        //   


	    if (usbd_audio_AltSet == 2) byte_per_sampl=6;
	    if (usbd_audio_AltSet == 1) byte_per_sampl=4;

	    num_samples = num_byte / byte_per_sampl;
	    num_remaining = (DMA1_Stream7->NDTR);
	    if ((num_remaining != num_remaining_old) || (PlayStart == 0))
	    {
	    	//    
	    	l = 0;
	    	for (k = 0; k < num_samples; k++)
	    	{
	    		if (spk_buffer_in == 0)
	    		{
	    			if (byte_per_sampl == 4)
	    			{
	    				Audio_buf_0[spk_index] = ((uint16_t)tmpbuf[l+1]<<8) | ((uint16_t)tmpbuf[l]);
	    				Audio_buf_0[spk_index+1] = 0;
	    				Audio_buf_0[spk_index+2] = ((uint16_t)tmpbuf[l+3]<<8) | ((uint16_t)tmpbuf[l+2]);
	    				Audio_buf_0[spk_index+3] = 0;
	    			}
	    			else
	    			{
	    				Audio_buf_0[spk_index] = ((uint16_t)tmpbuf[l+2]<<8) | ((uint16_t)tmpbuf[l+1]);
	    				Audio_buf_0[spk_index+1] = (uint16_t)tmpbuf[l];
	    				Audio_buf_0[spk_index+2] = ((uint16_t)tmpbuf[l+5]<<8) | ((uint16_t)tmpbuf[l+4]);
	    				Audio_buf_0[spk_index+3] = (uint16_t)tmpbuf[l+3]<<8;
	    			}
	    		}
	    		else
	    		{
	    			if (byte_per_sampl == 4)
	    			{
	    				Audio_buf_1[spk_index] = ((uint16_t)tmpbuf[l+1]<<8) | ((uint16_t)tmpbuf[l]);
	    				Audio_buf_1[spk_index+1] = 0;
	    				Audio_buf_1[spk_index+2] = ((uint16_t)tmpbuf[l+3]<<8) | ((uint16_t)tmpbuf[l+2]);
	    				Audio_buf_1[spk_index+3] = 0;
	    			}
	    			else
	    			{
	    				Audio_buf_1[spk_index] = ((uint16_t)tmpbuf[l+2]<<8) | ((uint16_t)tmpbuf[l+1]);
	    				Audio_buf_1[spk_index+1] = (uint16_t)tmpbuf[l];
	    				Audio_buf_1[spk_index+2] = ((uint16_t)tmpbuf[l+5]<<8) | ((uint16_t)tmpbuf[l+4]);
	    				Audio_buf_1[spk_index+3] = (uint16_t)tmpbuf[l+3]<<8;
	    			}
	    		}
	    		l += byte_per_sampl;
	    		spk_index += 4;
				if (spk_index >= AUDIO_BUFFER_SIZE)
				{
					spk_index = 0;
					spk_buffer_in = 1-spk_buffer_in;
					SOF_num = 0;
					/* Trigger the start of streaming only when spk_buffer_0 is full */
					if ((PlayStart == 0) && (spk_buffer_in == 1))
					{
						/* Enable start of Streaming */
						PlayStart = 1;
						FB_RATE = FB_NOM;
						AUDIO_OUT_fops.AudioCmd((uint32_t*)&Audio_buf_0,        /* Samples buffer pointer */
				                             	 AUDIO_BUFFER_SIZE,              /* Number of samples in Bytes */
				                             	 AUDIO_CMD_PLAY);                /* Command to be processed */
						//GPIO_ToggleBits(GPIOC,GPIO_Pin_1);
					}
				}
	    	}
	    	SOF_num++;
	    }
	    num_remaining_old = num_remaining;
        /* Toggle the frame index */
		DCD_EP_PrepareRx(pdev,
	                     AUDIO_OUT_EP,
	                    (uint8_t*)tmpbuf,
	                     USB_BUFFER_SIZE);

	    ((USB_OTG_CORE_HANDLE*)pdev)->dev.out_ep[epnum].even_odd_frame =\
		(((USB_OTG_CORE_HANDLE*)pdev)->dev.out_ep[epnum].even_odd_frame)? 0:1;
  	    out_stop = 1;
	}
return USBD_OK;
}

/**
  * @brief  usbd_audio_SOF
  *         Handles the SOF event (data buffer update and synchronization).
  * @param  pdev: instance
  * @param  epnum: endpoint number
  * @retval status
  */
static uint8_t  usbd_audio_SOF (void *pdev)
{     
	uint16_t gap;
	USB_OTG_DSTS_TypeDef  FS_DSTS;

//  
	if (((SOF_num == 15) && (Speed < 2)) || ((SOF_num == 7) && (Speed > 1)))
	{
		num_remaining = (DMA1_Stream7->NDTR);
		if (num_remaining != num_remaining_old)
		{
			gap = (AUDIO_BUFFER_SIZE-spk_index) + (AUDIO_BUFFER_SIZE-num_remaining);
			if (PlayStart)
			{
				if ((gap < (AUDIO_BUFFER_SIZE-20)) && (delta_num > (-FB_RATE_DELTA_NUM)))
				{
					GPIO_SetBits(GPIOD,GPIO_Pin_12);//GREEN ON
					FB_RATE -= FB_RATE_DELTA;
					delta_num--;
				}
				else if ((gap > (AUDIO_BUFFER_SIZE+20)) && (delta_num < FB_RATE_DELTA_NUM))
				{
					GPIO_SetBits(GPIOD,GPIO_Pin_14);//RED ON
					FB_RATE += FB_RATE_DELTA;
					delta_num++;
				}
				else
				{
					GPIO_ResetBits(GPIOD,GPIO_Pin_12);//GREEN OFF
					GPIO_ResetBits(GPIOD,GPIO_Pin_14);//RED OFF
				}
			}
		}
		else
		{
			FB_RATE = FB_NOM;
			delta_num = 0;
		}
	}
	if ((!flag))
	{
		FS_DSTS.d32 = USB_OTG_READ_REG32(&(((USB_OTG_CORE_HANDLE*)pdev)->regs.DREGS->DSTS));
		if (((FS_DSTS.b.soffn)&0x1) == dpid)
		{
			DCD_EP_Tx (pdev, AUDIO_IN_EP, (uint8_t *) &FB_RATE, 3);
			flag = 1;
		}
	}
	if (PlayStart == 1) GPIO_SetBits(GPIOD,GPIO_Pin_13);// 
	else GPIO_ResetBits(GPIOD,GPIO_Pin_13);
//	GPIO_ToggleBits(GPIOC,GPIO_Pin_1);
	//  OUT ,        
	if (out_stop == 0)
	{
		PlayStart = 0;
		spk_buffer_in = 0;
		spk_index = 0;
		delta_num = 0;
		//FB_RATE = FB_NOM;
		GPIO_ResetBits(GPIOD,GPIO_Pin_12);//GREEN OFF
		GPIO_ResetBits(GPIOD,GPIO_Pin_14);//RED OFF
//		GPIO_ToggleBits(GPIOC,GPIO_Pin_1);
	}
	out_stop = 0;
	return USBD_OK;
}

/**
  * @brief  usbd_audio_IN_Incplt
  *         Handles the iso out incomplete event.
  * @param  pdev: instance
  * @retval status
  */
static uint8_t  usbd_audio_IN_Incplt (void  *pdev)
{
	//This ISR is executed every time when IN token received with "wrong" PID. It's necessary
	//to flush IN EP (feedback EP), get parity value from DSTS, and store this info for SOF handler.
	//SOF handler should skip one frame with "wrong" PID and attempt a new transfer a frame later.

	USB_OTG_DSTS_TypeDef  FS_DSTS;
	FS_DSTS.d32 = USB_OTG_READ_REG32(&(((USB_OTG_CORE_HANDLE*)pdev)->regs.DREGS->DSTS));
	dpid=(FS_DSTS.b.soffn)&0x1;
	if (flag)
	{
		flag = 0;
		DCD_EP_Flush(pdev,AUDIO_IN_EP);
	}
	return USBD_OK;
}

/**
  * @brief  usbd_audio_OUT_Incplt
  *         Handles the iso out incomplete event.
  * @param  pdev: instance
  * @retval status
  */
static uint8_t  usbd_audio_OUT_Incplt (void  *pdev)
{
	return USBD_OK;
}

/******************************************************************************
     AUDIO Class requests management
******************************************************************************/
/**
  * @brief  AUDIO_Req_GetCurrent
  *         Handles the GET_CUR Audio control request.
  * @param  pdev: instance
  * @param  req: setup class request
  * @retval status
  */
static void AUDIO_Req_GetCurrent(void *pdev, USB_SETUP_REQ *req)
{  
    if (HIBYTE(req->wValue) == SAMPLING_FREQ_MUTE_CONTROL)//  SAMPLING_FREQ_CONTROL  MUTE
    {
    	if ((req->bmRequest & 0x0f) == ENDPOINT_REQUEST_TYPE)//    
    	{
        	/*AudioCtl[0] = (uint8_t)(Sampling_freq & 0xff);   //   
        	AudioCtl[1] = (uint8_t)((Sampling_freq>>8) & 0xff);
        	AudioCtl[2] = (uint8_t)((Sampling_freq>>16) & 0xff);*/
        	/* Send the current frequency state */
            USBD_CtlSendData (pdev,
            				 (uint8_t *)&Sampling_freq,
                              3);
    	}
    	else if (HIBYTE(req->wIndex) == AUDIO_OUT_STREAMING_CTRL)//   MUTE
    	{
    		AudioCtl[0] = mute;
    		/* Send the current mute state */
    	    USBD_CtlSendData (pdev,
    	                      AudioCtl,
    	                      1);
    	}
    }
}

/**
  * @brief  AUDIO_Req_SetCurrent
  *         Handles the SET_CUR Audio control request.
  * @param  pdev: instance
  * @param  req: setup class request
  * @retval status
  */
struct  usb_setup_req USB_SETUP_REQ2;

/**
  * @brief  AUDIO_Req_SetCurrent
  *         Handles the SET_CUR Audio control request.
  * @param  pdev: instance
  * @param  req: setup class request
  * @retval status
  */
static void AUDIO_Req_SetCurrent(void *pdev, USB_SETUP_REQ *req)
{
  if (req->wLength)
  {
    /* Prepare the reception of the buffer over EP0 */
    USBD_CtlPrepareRx (pdev,
                       AudioCtl,
                       req->wLength);

    /* Set the global variables indicating current request and its length
    to the function usbd_audio_EP0_RxReady() which will process the request */
    AudioCtlCmd = AUDIO_REQ_SET_CUR;     /* Set the request value */
    AudioCtlLen = req->wLength;          /* Set the request data length */
    AudioCtlUnit = HIBYTE(req->wIndex);  /* Set the request target unit */
    USB_SETUP_REQ2=*req;
  }
}

static uint8_t  usbd_audio_EP0_RxReady (void  *pdev)
  {
    uint32_t temp;
  	/* Check if an AudioControl request has been issued */
    if (AudioCtlCmd == AUDIO_REQ_SET_CUR)
    {/* In this driver, to simplify code, only SET_CUR request is managed */
      /* Check for which addressed unit the AudioControl request has been issued */
    	AUDIO_Req_SetCurrent2(&USB_SETUP_REQ2);
  	/*if (AudioCtlUnit == AUDIO_OUT_STREAMING_CTRL)
      {// In this driver, to simplify code, only one unit is manage
        // Call the audio interface mute function
        AUDIO_OUT_fops.MuteCtl(AudioCtl[0]);


      }*/
    /* Reset the AudioCtlCmd variable to prevent re-entering this function */
    AudioCtlCmd = 0;
    AudioCtlLen = 0;
    }
/*    if (AudioCtlCmd == AUDIO_REQ_GET_CUR)
      {
    	  temp=48000;
       AudioCtl[0]=(uint8_t)(temp&0xff);
       AudioCtl[1]=(uint8_t)((temp>>8)&0xff);
       AudioCtl[2]=(uint8_t)((temp>>16)&0xff);
      //Reset the AudioCtlCmd variable to prevent re-entering this function
      AudioCtlCmd = 0;
      AudioCtlLen = 0;
      }
*/
    return USBD_OK;
  }

static void AUDIO_Req_SetCurrent2(USB_SETUP_REQ *req)
{ 
	if (req->wLength)
	{
		/* Prepare the reception of the buffer over EP0 */
    	//USBD_CtlPrepareRx (pdev,\
                       	   AudioCtl,\
                       	   req->wLength);
    
    	if (HIBYTE(req->wValue) == SAMPLING_FREQ_MUTE_CONTROL)//  SAMPLING_FREQ_CONTROL  MUTE
    	{
    		if ((req->bmRequest & 0x0f) == ENDPOINT_REQUEST_TYPE)//    
    		{
    			//  
    			Sampling_freq = (*(uint32_t *)&AudioCtl[0]) & 0x00ffffff;
    			//     FB_RATE   
    			switch (Sampling_freq)
    			{
    				case 44100 :
    				{
    					FB_NOM = (44 << 14) + (1 << 14) / 10;
    					//SPI3->I2SPR = 0x0004;//  I2S = 4
    					 SetI2SMCLK(Sampling_freq);

    					mult=1;
    				};break;
    				case 48000 :
    				{
    					FB_NOM = 48 << 14;
    					//SPI3->I2SPR = 0x0004;//  I2S = 4
    					SetI2SMCLK(Sampling_freq);

    					mult=1;
    				};break;
    				case 88200 :
    				{
    					FB_NOM = (88 << 14) + (2 << 14) / 10;
    					//SPI3->I2SPR = 0x0002;//  I2S = 2
    					SetI2SMCLK(Sampling_freq);

    					mult=2;
    				};break;
    				case 96000 :
    				{
    					FB_NOM = 96 << 14;
    					//SPI3->I2SPR = 0x0002;//  I2S = 2
    					SetI2SMCLK(Sampling_freq);

    					mult=2;
    				};break;
    				case 176400 :
    				{
    					FB_NOM = 176 << 14 + (4 << 14) / 10;
    					//SPI3->I2SPR = 0x0002;//  I2S = 2
    					SetI2SMCLK(Sampling_freq);

    					mult=4;
    				};break;
    				case 192000 :
    				{
    					FB_NOM = 192 << 14;
    					//SPI3->I2SPR = 0x0002;//  I2S = 2
    					SetI2SMCLK(Sampling_freq);

    					mult=4;
    				};break;
    			}
    			FB_RATE = FB_NOM;
    			delta_num = 0;
    			PlayStart = 0;
    			spk_buffer_in = 0;
    			spk_buffer_out = 0;
    			spk_index = 0;
    			delta_num = 0;
    			old_gap = AUDIO_BUFFER_SIZE;
    		}
    		else if (HIBYTE(req->wIndex) == AUDIO_OUT_STREAMING_CTRL)//   MUTE
    		{
    			mute = AudioCtl[0];
//    			if (mute) GPIO_SetBits(GPIOB,GPIO_Pin_4);
//    			else GPIO_ResetBits(GPIOB,GPIO_Pin_4);
    		}
    	}
	}
}
/**
  * @brief  USBD_audio_GetCfgDesc 
  *         Returns configuration descriptor.
  * @param  speed : current device speed
  * @param  length : pointer data length
  * @retval pointer to descriptor buffer
  */
static uint8_t  *USBD_audio_GetCfgDesc (uint8_t speed, uint16_t *length)
{
	*length = sizeof (usbd_audio_CfgDesc);
	return usbd_audio_CfgDesc;
}

static uint8_t  *USBD_audio_GetOtherSpeedCfgDesc ( uint8_t speed, uint16_t *length)
{ if (speed == USB_OTG_SPEED_FULL)
*length = sizeof (usbd_audio_FSCfgDesc);
return usbd_audio_FSCfgDesc;
}

void Audio_Handler(void)
{
    /* Disable the I2S DMA Stream*/
    //DMA_Cmd(DMA1_Stream7, DISABLE);
    DMA1_Stream7->CR &= ~(uint32_t)DMA_SxCR_EN;

    /* Clear the Interrupt flag */
    //DMA_ClearFlag(DMA1_Stream7, DMA_FLAG_TCIF7 |DMA_FLAG_HTIF7 | DMA_FLAG_FEIF7 | DMA_FLAG_TEIF7);
    DMA1->HIFCR = (uint32_t)((DMA_FLAG_TCIF7 | DMA_FLAG_HTIF7 | DMA_FLAG_FEIF7 |
    		                  DMA_FLAG_TEIF7) & RESERVED_MASK);

    if (PlayStart == 1)
    {
//    	GPIO_ToggleBits(GPIOC,GPIO_Pin_1);
    	if (spk_buffer_out == 0)
    	{
    		AUDIO_OUT_fops.AudioCmd((uint32_t*)Audio_buf_1,         /* Samples buffer pointer */
                               	    AUDIO_BUFFER_SIZE,               /* Number of samples in Bytes */
                               	    AUDIO_CMD_PLAY);                 /* Command to be processed */
    	}
    	else
    	{
    		AUDIO_OUT_fops.AudioCmd((uint32_t*)Audio_buf_0,         /* Samples buffer pointer */
                               	    AUDIO_BUFFER_SIZE,               /* Number of samples in Bytes */
                               	    AUDIO_CMD_PLAY);                 /* Command to be processed */
    	}
    	spk_buffer_out = 1-spk_buffer_out;
    }
    else
    {
		AUDIO_OUT_fops.AudioCmd((uint32_t*)Audio_buf_0,         /* Samples buffer pointer */
                           	    AUDIO_BUFFER_SIZE,               /* Number of samples in Bytes */
                           	    AUDIO_CMD_STOP);                 /* Command to be processed */
		spk_buffer_out = 0;
	}
}
/**
  * @}
  */ 

/**
  * @}
  */ 

/**
  * @}
  */ 

/******************* (C) COPYRIGHT 2011 STMicroelectronics *****END OF FILE****/
