import scikits.audiolab as au
import numpy as np
import time
import random
import pylab as plt
import sys

rate = 44100 # sampling rate
freq = 90 # tone frequency
harmLevel = 0.02 # 3d harmonic level
secs = 3 # playing back duration for each signal
level = 0.7 # playing back level
shiftA = 1.0 # signal A phase shift, in PI
shiftB = 0.0 # signal B phase shift, in PI
log = "log-" + str(harmLevel) + ".txt" # trials are logged here

def getSamples(shift):
    t = np.linspace(0.0, secs, rate * secs + 1)
    tone = np.sin(2.0 * np.pi * freq * t)
    harm = np.sin(6.0 * np.pi * freq * t + np.pi * shift) * harmLevel
    return fade(tone + harm)

def fade(data):
    l = rate / 10 # 0.1 seconds
    win = np.linspace(0.0, 1.0, l)
    for i in range(0, l):
        data[i] = data[i] * win[i] 
        data[-i-1] = data[-i-1] * win[i] 
    return data

def plotAB():
    start = rate / 5 # skip fading
    stop = start + rate * 4 / freq # four periods 
    lim = 1.1 + harmLevel
    
    plt.subplot(2, 1, 1)
    plt.plot(getSamples(shiftA)[start : stop])
    plt.title("shift is pi*" + str(shiftA))
    plt.ylim(-lim, lim)
    
    plt.subplot(2, 1, 2)
    plt.plot(getSamples(shiftB)[start : stop])
    plt.title("shift is pi*" + str(shiftB))
    plt.ylim(-lim, lim)
    
    plt.show()
    
def playSingle(s):
    au.play(getSamples(s) * level/(1.01 + harmLevel), rate)
    
def playAB():
    playSingle(shiftA)
    time.sleep(0.1)
    playSingle(shiftB)

def playAX():
    r = shiftA # AA
    areDiff = random.randint(0, 1) == 1
    if areDiff: r = shiftB # AB
    playSingle(shiftA)
    time.sleep(0.2)
    playSingle(r)
    processAnswer(areDiff)
    
def printTrialHelp():
    print("")
    print("Type in your opinion:")
    print("")
    print("a - second fragment is A also")
    print("b - second fragment is B")
    print("<anything else> - ignore this trial at all")
    print("")
    
def processAnswer(wasDiff):
    printTrialHelp()
    answer = raw_input("Your opinion: ").lower()
    if answer == "a": logAnswer(wasDiff, False)
    elif answer == "b": logAnswer(wasDiff, True)
    else: print("the trial was ignored..")

def logAnswer(reality, opinion):
    now = time.strftime("%Y-%H:%M:%S", time.localtime())
    if reality: re = "A and B"
    else: re = "A and A"
    if opinion: op = "A and B"
    else: op = "A and A"
    out = now + " - reality: " + re + ", opinion: " + op
    if re != op: out = out + ", Error"
    f = open(log, "a") # in text mode
    f.writelines(out + "\n")
    f.close()

def printHelp(me):
    print("Usage:")
    print("")
    print("python " + me + " <command>")
    print("")
    print("where <command> is:")
    print("")
    print("plot - plots signals A and B")
    print("ab - plays signals A and B")
    print("ax - plays signals A and X, asks your opinion about difference")
    print("<any other> - prints this help ")
    print("")
    
def main(args):
    me = args[0]
    if len(args) != 2:
        printHelp(me)
        return
    cmd = args[1]
    if cmd == "plot": 
        plotAB()
    elif cmd.lower() == "ab": 
        playAB()
    elif cmd.lower() == "ax": 
        playAX()
    else: 
        printHelp(me)


if __name__ == '__main__':
    main(sys.argv)
